import { useEffect, useState } from "react";
import { useMap } from "react-leaflet";
import L from "leaflet";
import { HotZone } from "@shared/schema";

interface HotZonesLayerProps {
  hotZones: HotZone[];
  visible: boolean;
  onZoneSelect?: (zone: HotZone) => void;
  editable?: boolean; // Allow operators to create/edit zones
}

export default function HotZonesLayer({ 
  hotZones, 
  visible, 
  onZoneSelect,
  editable = false 
}: HotZonesLayerProps) {
  const map = useMap();
  const [isDrawing, setIsDrawing] = useState(false);

  useEffect(() => {
    if (!visible) return;

    const zoneOverlays: L.Rectangle[] = [];

    hotZones.forEach((zone) => {
      if (!zone.active) return;

      // Get stress level color
      const getStressColor = (level: number) => {
        if (level >= 8) return '#dc2626'; // red - critical
        if (level >= 6) return '#ea580c'; // orange - high
        if (level >= 4) return '#d97706'; // amber - medium
        return '#059669'; // emerald - low
      };

      const bounds = L.latLngBounds(
        [zone.bounds.south, zone.bounds.west],
        [zone.bounds.north, zone.bounds.east]
      );

      const overlay = L.rectangle(bounds, {
        color: getStressColor(zone.stressLevel),
        weight: 3,
        opacity: 0.8,
        fillOpacity: 0.2,
        className: `hot-zone-${zone.stressLevel}`
      });

      // Create popup content with safe HTML
      const popupContent = L.DomUtil.create('div', 'p-3 space-y-2');
      
      // Header with color indicator and zone name
      const header = L.DomUtil.create('div', 'font-bold text-sm flex items-center gap-2');
      const colorDiv = L.DomUtil.create('div', 'w-3 h-3 rounded');
      colorDiv.style.backgroundColor = getStressColor(zone.stressLevel);
      header.appendChild(colorDiv);
      header.appendChild(document.createTextNode(zone.name));
      
      // Details section
      const details = L.DomUtil.create('div', 'text-xs space-y-1');
      details.innerHTML = `
        <div><strong>Stress Level:</strong> ${zone.stressLevel}/10</div>
        <div><strong>Priority Bonus:</strong> +${zone.priority}%</div>
        <div><strong>Status:</strong> ${zone.active ? 'Active' : 'Inactive'}</div>
      `;
      
      popupContent.appendChild(header);
      popupContent.appendChild(details);
      
      if (editable) {
        const buttonContainer = L.DomUtil.create('div', 'pt-2 border-t');
        
        const editButton = L.DomUtil.create('button', 'text-xs bg-blue-500 text-white px-2 py-1 rounded hover:bg-blue-600');
        editButton.textContent = 'Edit Zone';
        editButton.onclick = () => editHotZone(zone.id);
        
        const deleteButton = L.DomUtil.create('button', 'text-xs bg-red-500 text-white px-2 py-1 rounded hover:bg-red-600 ml-2');
        deleteButton.textContent = 'Delete';
        deleteButton.onclick = () => deleteHotZone(zone.id);
        
        buttonContainer.appendChild(editButton);
        buttonContainer.appendChild(deleteButton);
        popupContent.appendChild(buttonContainer);
      }
      
      overlay.bindPopup(popupContent);

      // Click handler for zone selection
      overlay.on('click', () => {
        if (onZoneSelect) {
          onZoneSelect(zone);
        }
      });

      overlay.addTo(map);
      zoneOverlays.push(overlay);
    });

    // Add drawing capability for operators
    if (editable) {
      const addDrawingControl = () => {
        const drawButton = L.control({ position: 'topright' });
        drawButton.onAdd = () => {
          const div = L.DomUtil.create('div', 'leaflet-control-custom');
          div.innerHTML = `
            <button 
              class="bg-primary text-primary-foreground px-3 py-2 rounded shadow hover:bg-primary/90"
              title="Draw Hot Zone"
            >
              ⬛ Draw Zone
            </button>
          `;
          
          div.onclick = () => {
            setIsDrawing(!isDrawing);
            // TODO: Implement drawing mode with Leaflet.draw or custom implementation
          };
          
          return div;
        };
        
        drawButton.addTo(map);
        return drawButton;
      };

      const drawControl = addDrawingControl();
      
      return () => {
        zoneOverlays.forEach(overlay => map.removeLayer(overlay));
        map.removeControl(drawControl);
      };
    }

    // Cleanup function
    return () => {
      zoneOverlays.forEach(overlay => {
        map.removeLayer(overlay);
      });
    };
  }, [map, hotZones, visible, editable, onZoneSelect, isDrawing]);

  return null;
}

// Safe functions for hot zone editing (no global window pollution)
const editHotZone = (zoneId: string) => {
  console.log('Edit hot zone:', zoneId);
  // TODO: Implement hot zone editing modal
};

const deleteHotZone = (zoneId: string) => {
  console.log('Delete hot zone:', zoneId);
  // TODO: Implement hot zone deletion
};