import { useState, useMemo } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Slider } from "@/components/ui/slider";
import { Badge } from "@/components/ui/badge";
import { Separator } from "@/components/ui/separator";
import { Progress } from "@/components/ui/progress";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { 
  Target, 
  Zap, 
  Battery, 
  Users, 
  MapPin, 
  AlertTriangle,
  CheckCircle,
  Clock,
  DollarSign
} from "lucide-react";
import { EVScoring, WeightConfig, DispatchPlan, HotZone } from "@shared/schema";

interface InviteEnginePanelProps {
  vehicleScores: EVScoring[];
  hotZones: HotZone[];
  onWeightChange: (weights: WeightConfig) => void;
  onDispatchPlan: (plan: DispatchPlan) => void;
  onInviteVehicles: (vehicleIds: string[]) => void;
  className?: string;
}

export default function InviteEnginePanel({
  vehicleScores,
  hotZones,
  onWeightChange,
  onDispatchPlan,
  onInviteVehicles,
  className = ""
}: InviteEnginePanelProps) {
  const [targetKW, setTargetKW] = useState(500);
  const [weights, setWeights] = useState<WeightConfig>({
    availability: 0.2,
    socBuffer: 0.15,
    batteryHealth: 0.15,
    locationPriority: 0.15,
    fleetBias: 0.1,
    fairnessRotation: 0.1,
    technicalReadiness: 0.1,
    consentRequired: 0.05,
  });
  const [selectedVehicles, setSelectedVehicles] = useState<string[]>([]);
  const [activeHotZones, setActiveHotZones] = useState<string[]>(
    hotZones.filter(z => z.active).map(z => z.id)
  );

  // Calculate weighted scores and sort vehicles
  const rankedVehicles = useMemo(() => {
    return vehicleScores
      .map(vehicle => {
        const factors = vehicle.factors;
        const weightedScore = 
          factors.availability * weights.availability +
          factors.socBuffer * weights.socBuffer +
          (factors.batteryHealth / 100) * weights.batteryHealth +
          factors.locationPriority * weights.locationPriority +
          factors.fleetBonus * weights.fleetBias +
          (factors.technicalReadiness ? 1 : 0) * weights.technicalReadiness +
          (factors.consent ? 1 : 0) * weights.consentRequired;

        return {
          ...vehicle,
          weightedScore: Math.min(100, weightedScore * 100)
        };
      })
      .sort((a, b) => b.weightedScore - a.weightedScore);
  }, [vehicleScores, weights]);

  // Calculate dispatch plan
  const currentPlan = useMemo(() => {
    const selectedData = rankedVehicles.filter(v => selectedVehicles.includes(v.vehicleId));
    const totalAvailable = selectedData.reduce((sum, v) => sum + v.maxDischargeKW, 0);
    
    return {
      id: `plan-${Date.now()}`,
      targetKW,
      selectedVehicles,
      totalAvailableKW: totalAvailable,
      shortfall: targetKW - totalAvailable,
      estimatedDuration: selectedData.length > 0 ? 
        Math.min(...selectedData.map(v => (v.surplusKWh / v.maxDischargeKW) * 60)) : 0,
      created: new Date().toISOString()
    };
  }, [selectedVehicles, targetKW, rankedVehicles]);

  const handleWeightChange = (factor: keyof WeightConfig, value: number[]) => {
    const newWeights = { ...weights, [factor]: value[0] };
    setWeights(newWeights);
    onWeightChange(newWeights);
  };

  const toggleVehicleSelection = (vehicleId: string) => {
    setSelectedVehicles(prev => 
      prev.includes(vehicleId) 
        ? prev.filter(id => id !== vehicleId)
        : [...prev, vehicleId]
    );
  };

  const autoSelectOptimal = () => {
    let totalPower = 0;
    const selected: string[] = [];
    
    for (const vehicle of rankedVehicles) {
      if (totalPower >= targetKW) break;
      if (!vehicle.factors.consent) continue;
      
      selected.push(vehicle.vehicleId);
      totalPower += vehicle.maxDischargeKW;
    }
    
    setSelectedVehicles(selected);
  };

  const clearSelection = () => {
    setSelectedVehicles([]);
  };

  const sendInvites = () => {
    onInviteVehicles(selectedVehicles);
    onDispatchPlan(currentPlan);
  };

  return (
    <Card className={`h-full ${className}`}>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Target className="w-5 h-5" />
          Invite Engine
        </CardTitle>
      </CardHeader>
      <CardContent className="space-y-4 max-h-[calc(100vh-200px)] overflow-y-auto">
        {/* Target Dispatch Control */}
        <div className="space-y-2">
          <Label htmlFor="target-kw">Target Power (kW)</Label>
          <div className="flex items-center gap-2">
            <Input
              id="target-kw"
              type="number"
              value={targetKW}
              onChange={(e) => setTargetKW(Number(e.target.value))}
              className="w-24"
              data-testid="input-target-kw"
            />
            <Button 
              variant="outline" 
              size="sm"
              onClick={autoSelectOptimal}
              data-testid="button-auto-select"
            >
              Auto Select
            </Button>
            <Button 
              variant="outline" 
              size="sm"
              onClick={clearSelection}
              data-testid="button-clear-selection"
            >
              Clear
            </Button>
          </div>
        </div>

        <Separator />

        {/* Weight Sliders */}
        <div className="space-y-3">
          <h4 className="font-semibold text-sm">Scoring Weights</h4>
          
          {Object.entries(weights).map(([factor, value]) => (
            <div key={factor} className="space-y-1">
              <div className="flex justify-between text-xs">
                <span className="capitalize">{factor.replace(/([A-Z])/g, ' $1')}</span>
                <span>{(value * 100).toFixed(0)}%</span>
              </div>
              <Slider
                value={[value]}
                onValueChange={(val) => handleWeightChange(factor as keyof WeightConfig, val)}
                max={1}
                min={0}
                step={0.05}
                className="w-full"
                data-testid={`slider-${factor}`}
              />
            </div>
          ))}
        </div>

        <Separator />

        {/* Hot Zone Selector */}
        <div className="space-y-2">
          <h4 className="font-semibold text-sm">Active Hot Zones</h4>
          <div className="flex flex-wrap gap-1">
            {hotZones.map(zone => (
              <Badge
                key={zone.id}
                variant={activeHotZones.includes(zone.id) ? "default" : "outline"}
                className="cursor-pointer text-xs"
                onClick={() => setActiveHotZones(prev => 
                  prev.includes(zone.id)
                    ? prev.filter(id => id !== zone.id)
                    : [...prev, zone.id]
                )}
                data-testid={`zone-toggle-${zone.id}`}
              >
                🔥 {zone.name}
              </Badge>
            ))}
          </div>
        </div>

        <Separator />

        {/* Ranked Candidate List */}
        <div className="space-y-2">
          <h4 className="font-semibold text-sm">Ranked Candidates</h4>
          <div className="space-y-1 max-h-40 overflow-y-auto">
            {rankedVehicles.slice(0, 10).map((vehicle, index) => (
              <div
                key={vehicle.vehicleId}
                className={`p-2 rounded border cursor-pointer transition-colors ${
                  selectedVehicles.includes(vehicle.vehicleId)
                    ? 'bg-primary/10 border-primary'
                    : 'hover:bg-muted'
                } ${!vehicle.factors.consent ? 'opacity-50' : ''}`}
                onClick={() => vehicle.factors.consent && toggleVehicleSelection(vehicle.vehicleId)}
                data-testid={`vehicle-candidate-${vehicle.vehicleId}`}
              >
                <div className="flex items-center justify-between">
                  <div className="flex items-center gap-2">
                    <span className="text-xs font-mono">#{index + 1}</span>
                    <span className="text-sm font-medium">{vehicle.vehicleId}</span>
                    {!vehicle.factors.consent && (
                      <AlertTriangle className="w-3 h-3 text-yellow-500" />
                    )}
                    {selectedVehicles.includes(vehicle.vehicleId) && (
                      <CheckCircle className="w-3 h-3 text-green-500" />
                    )}
                  </div>
                  <div className="text-right text-xs">
                    <div>Score: {vehicle.weightedScore.toFixed(1)}</div>
                    <div>{vehicle.maxDischargeKW}kW</div>
                  </div>
                </div>
                <div className="mt-1">
                  <Progress value={vehicle.weightedScore} className="h-1" />
                </div>
              </div>
            ))}
          </div>
        </div>

        <Separator />

        {/* Dispatch Plan Summary */}
        <div className="space-y-2">
          <h4 className="font-semibold text-sm">Dispatch Plan Preview</h4>
          <div className="grid grid-cols-2 gap-2 text-xs">
            <div className="flex items-center gap-1">
              <Target className="w-3 h-3" />
              <span>Target: {targetKW} kW</span>
            </div>
            <div className="flex items-center gap-1">
              <Zap className="w-3 h-3" />
              <span>Available: {currentPlan.totalAvailableKW.toFixed(1)} kW</span>
            </div>
            <div className="flex items-center gap-1">
              <Users className="w-3 h-3" />
              <span>Vehicles: {selectedVehicles.length}</span>
            </div>
            <div className="flex items-center gap-1">
              <Clock className="w-3 h-3" />
              <span>Duration: {currentPlan.estimatedDuration.toFixed(0)}min</span>
            </div>
          </div>
          
          {/* Shortfall indicator */}
          {currentPlan.shortfall !== 0 && (
            <div className={`p-2 rounded text-xs ${
              currentPlan.shortfall > 0 
                ? 'bg-red-100 text-red-800 dark:bg-red-900/20 dark:text-red-400'
                : 'bg-green-100 text-green-800 dark:bg-green-900/20 dark:text-green-400'
            }`}>
              {currentPlan.shortfall > 0 
                ? `⚠️ Shortfall: ${currentPlan.shortfall.toFixed(1)} kW`
                : `✅ Surplus: ${Math.abs(currentPlan.shortfall).toFixed(1)} kW`
              }
            </div>
          )}
        </div>

        {/* Action Buttons */}
        <div className="flex gap-2">
          <Button
            onClick={sendInvites}
            disabled={selectedVehicles.length === 0}
            className="flex-1"
            data-testid="button-send-invites"
          >
            <DollarSign className="w-4 h-4 mr-1" />
            Send Invites ({selectedVehicles.length})
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}