import { useState } from "react";
import { X, Eye, Clock, Bell } from "lucide-react";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { cn } from "@/lib/utils";
import type { SelectPing } from "@shared/schema";

interface PingBannerProps {
  ping: SelectPing;
  onView?: (ping: SelectPing) => void;
  onSnooze?: (ping: SelectPing, minutes: number) => void;
  onDismiss?: (ping: SelectPing) => void;
  className?: string;
}

export function PingBanner({ 
  ping, 
  onView, 
  onSnooze, 
  onDismiss, 
  className 
}: PingBannerProps) {
  const [isExpanded, setIsExpanded] = useState(false);

  const getPriorityColor = (priority: string) => {
    switch (priority.toLowerCase()) {
      case "urgent":
        return "bg-red-500/10 border-red-500/50 text-red-700 dark:text-red-400";
      case "high":
        return "bg-orange-500/10 border-orange-500/50 text-orange-700 dark:text-orange-400";
      case "normal":
        return "bg-blue-500/10 border-blue-500/50 text-blue-700 dark:text-blue-400";
      default:
        return "bg-gray-500/10 border-gray-500/50 text-gray-700 dark:text-gray-400";
    }
  };

  const getTypeIcon = (type: string) => {
    switch (type.toLowerCase()) {
      case "grid_invite":
        return "🔌";
      case "dispatch_request":
        return "⚡";
      case "emergency_alert":
        return "🚨";
      default:
        return "📢";
    }
  };

  const formatTimeAgo = (timestamp: string) => {
    const now = new Date();
    const sentTime = new Date(timestamp);
    const diffMinutes = Math.floor((now.getTime() - sentTime.getTime()) / 1000 / 60);
    
    if (diffMinutes < 1) return "just now";
    if (diffMinutes < 60) return `${diffMinutes}m ago`;
    if (diffMinutes < 1440) return `${Math.floor(diffMinutes / 60)}h ago`;
    return `${Math.floor(diffMinutes / 1440)}d ago`;
  };

  return (
    <div 
      className={cn(
        "border rounded-lg p-3 shadow-sm transition-all duration-200",
        getPriorityColor(ping.priority),
        isExpanded ? "ring-1 ring-primary/20" : "",
        className
      )}
      data-testid={`ping-banner-${ping.id}`}
    >
      {/* Header Row */}
      <div className="flex items-start justify-between gap-2">
        <div className="flex items-start gap-2 flex-1 min-w-0">
          <span className="text-lg flex-shrink-0" role="img" aria-label={ping.type}>
            {getTypeIcon(ping.type)}
          </span>
          
          <div className="flex-1 min-w-0">
            <div className="flex items-center gap-2 mb-1">
              <h4 className="font-medium text-sm truncate">
                {ping.title}
              </h4>
              <Badge 
                variant="secondary" 
                className="text-xs py-0 px-1 h-5"
                data-testid={`badge-priority-${ping.priority}`}
              >
                {ping.priority.toUpperCase()}
              </Badge>
            </div>
            
            <p className={cn(
              "text-sm text-muted-foreground",
              isExpanded ? "" : "line-clamp-1"
            )}>
              {ping.message}
            </p>
            
            <div className="flex items-center gap-2 mt-1 text-xs text-muted-foreground">
              <Clock className="h-3 w-3" />
              <span>{formatTimeAgo(ping.sentAt)}</span>
              {ping.region && (
                <>
                  <span>•</span>
                  <span>{ping.region}</span>
                </>
              )}
            </div>
          </div>
        </div>

        {/* Actions */}
        <div className="flex items-center gap-1 flex-shrink-0">
          <Button
            variant="ghost"
            size="sm"
            onClick={() => setIsExpanded(!isExpanded)}
            className="h-8 w-8 p-0"
            data-testid="button-expand-ping"
          >
            <Bell className="h-3 w-3" />
          </Button>
          
          {onDismiss && (
            <Button
              variant="ghost"
              size="sm"
              onClick={() => onDismiss(ping)}
              className="h-8 w-8 p-0 hover:bg-destructive/10 hover:text-destructive"
              data-testid="button-dismiss-ping"
            >
              <X className="h-3 w-3" />
            </Button>
          )}
        </div>
      </div>

      {/* Expanded Actions */}
      {isExpanded && (
        <div className="flex gap-2 mt-3 pt-2 border-t border-current/10">
          {onView && (
            <Button
              variant="default"
              size="sm"
              onClick={() => onView(ping)}
              className="flex-1 gap-2"
              data-testid="button-view-ping"
            >
              <Eye className="h-3 w-3" />
              View Details
            </Button>
          )}
          
          {onSnooze && (
            <Button
              variant="outline"
              size="sm"
              onClick={() => onSnooze(ping, 15)}
              className="gap-2"
              data-testid="button-snooze-ping"
            >
              <Clock className="h-3 w-3" />
              15m
            </Button>
          )}
        </div>
      )}

      {/* Payload Preview (if expanded and has payload) */}
      {isExpanded && ping.payload && (
        <div className="mt-2 pt-2 border-t border-current/10">
          <details className="text-xs">
            <summary className="cursor-pointer text-muted-foreground">
              Additional Details
            </summary>
            <pre className="mt-1 p-2 bg-muted/50 rounded text-xs overflow-x-auto">
              {JSON.stringify(JSON.parse(ping.payload), null, 2)}
            </pre>
          </details>
        </div>
      )}
    </div>
  );
}

interface PingBannerListProps {
  pings: SelectPing[];
  onView?: (ping: SelectPing) => void;
  onSnooze?: (ping: SelectPing, minutes: number) => void;
  onDismiss?: (ping: SelectPing) => void;
  maxVisible?: number;
  className?: string;
}

export function PingBannerList({ 
  pings, 
  onView, 
  onSnooze, 
  onDismiss, 
  maxVisible = 3,
  className 
}: PingBannerListProps) {
  const [showAll, setShowAll] = useState(false);
  
  const visiblePings = showAll ? pings : pings.slice(0, maxVisible);
  const remainingCount = pings.length - maxVisible;

  if (pings.length === 0) return null;

  return (
    <div 
      className={cn("space-y-2", className)}
      data-testid="ping-banner-list"
    >
      {visiblePings.map((ping) => (
        <PingBanner
          key={ping.id}
          ping={ping}
          onView={onView}
          onSnooze={onSnooze}
          onDismiss={onDismiss}
        />
      ))}
      
      {!showAll && remainingCount > 0 && (
        <Button
          variant="outline"
          size="sm"
          onClick={() => setShowAll(true)}
          className="w-full"
          data-testid="button-show-all-pings"
        >
          Show {remainingCount} more notification{remainingCount !== 1 ? 's' : ''}
        </Button>
      )}
      
      {showAll && pings.length > maxVisible && (
        <Button
          variant="outline"
          size="sm"
          onClick={() => setShowAll(false)}
          className="w-full"
          data-testid="button-collapse-pings"
        >
          Show less
        </Button>
      )}
    </div>
  );
}