import { useState } from "react";
import { motion } from "framer-motion";
import { useLocation } from "wouter";
import VehicleMap, { Vehicle } from "./VehicleMap";
import VehicleInfoPanel from "./VehicleInfoPanel";
import { ThemeToggle } from "./ThemeToggle";
import { MobileShell } from "./MobileShell";
import { FiltersSheet } from "./FiltersSheet";
import { PingBannerList } from "./PingBanner";
import { AppSidebar } from "./app-sidebar";
import { SidebarProvider, SidebarTrigger } from "@/components/ui/sidebar";
import { NotificationProvider, useNotifications } from "@/hooks/useNotifications";
import { useIsMobile } from "@/hooks/useMobile";
import { V2GCall, RegionStat, GridInvitation, EVScoring, HotZone, EconomicsData } from "@shared/schema";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { 
  Zap, 
  Car, 
  Battery, 
  TrendingUp, 
  RefreshCw,
  AlertCircle,
  LayoutDashboard,
  BarChart3,
  Calculator,
  Settings2,
  Eye,
  EyeOff,
  Filter,
  Package
} from "lucide-react";
import GridRequirements from "../pages/GridRequirements";
import VehiclesDashboard from "../pages/VehiclesDashboard";
import ChargingAnalytics from "../pages/ChargingAnalytics";
import IncentiveSimulation from "../pages/IncentiveSimulation";
import Settings from "../pages/Settings";
import CriteriaManagement from "../pages/CriteriaManagement";
import AssetManagement from "../pages/AssetManagement";

// TODO: remove mock functionality
const mockVehicles: Vehicle[] = [
  { id: 1, name: "EV-101", lat: 1.3521, lng: 103.8198, soc: 80, status: "Idle", battery: 60, fleet: false },
  { id: 2, name: "EV-202", lat: 1.3621, lng: 103.8298, soc: 45, status: "Discharging", battery: 120, fleet: true },
  { id: 3, name: "EV-303", lat: 1.3721, lng: 103.8098, soc: 90, status: "Charging", battery: 75, fleet: false },
  { id: 4, name: "EV-404", lat: 1.3421, lng: 103.8398, soc: 25, status: "Offline", battery: 85, fleet: true },
  { id: 5, name: "EV-505", lat: 1.3321, lng: 103.8498, soc: 65, status: "Charging", battery: 95, fleet: false },
];

const mockDemandData = [
  { time: "10:00", demand: 120 },
  { time: "12:00", demand: 180 },
  { time: "14:00", demand: 150 },
  { time: "16:00", demand: 200 },
  { time: "18:00", demand: 250 },
  { time: "20:00", demand: 220 },
];

// Mock vehicle scoring data
const mockVehicleScores: EVScoring[] = [
  {
    vehicleId: "EV-101",
    eligibilityScore: 85.5,
    factors: {
      availability: 0.9,
      socBuffer: 0.8,
      consent: true,
      batteryHealth: 92,
      batteryFatigue: 0.15,
      incentiveClass: "premium",
      technicalReadiness: true,
      locationPriority: 0.7,
      fleetBonus: 0.0
    },
    surplusKWh: 35.2,
    maxDischargeKW: 45,
    lastRotation: new Date(Date.now() - 24 * 60 * 60 * 1000).toISOString(),
    weeklyDischarges: 3
  },
  {
    vehicleId: "EV-202",
    eligibilityScore: 92.3,
    factors: {
      availability: 0.95,
      socBuffer: 0.85,
      consent: true,
      batteryHealth: 89,
      batteryFatigue: 0.2,
      incentiveClass: "fleet",
      technicalReadiness: true,
      locationPriority: 0.9,
      fleetBonus: 0.2
    },
    surplusKWh: 67.8,
    maxDischargeKW: 85,
    lastRotation: new Date(Date.now() - 48 * 60 * 60 * 1000).toISOString(),
    weeklyDischarges: 2
  },
  {
    vehicleId: "EV-303",
    eligibilityScore: 78.1,
    factors: {
      availability: 0.7,
      socBuffer: 0.9,
      consent: true,
      batteryHealth: 95,
      batteryFatigue: 0.1,
      incentiveClass: "standard",
      technicalReadiness: true,
      locationPriority: 0.5,
      fleetBonus: 0.0
    },
    surplusKWh: 52.1,
    maxDischargeKW: 65,
    lastRotation: new Date(Date.now() - 12 * 60 * 60 * 1000).toISOString(),
    weeklyDischarges: 5
  },
  {
    vehicleId: "EV-404",
    eligibilityScore: 45.2,
    factors: {
      availability: 0.3,
      socBuffer: 0.2,
      consent: false,
      batteryHealth: 78,
      batteryFatigue: 0.4,
      incentiveClass: "basic",
      technicalReadiness: false,
      locationPriority: 0.3,
      fleetBonus: 0.1
    },
    surplusKWh: 12.3,
    maxDischargeKW: 30,
    lastRotation: new Date(Date.now() - 6 * 60 * 60 * 1000).toISOString(),
    weeklyDischarges: 8
  },
  {
    vehicleId: "EV-505",
    eligibilityScore: 89.7,
    factors: {
      availability: 0.85,
      socBuffer: 0.95,
      consent: true,
      batteryHealth: 96,
      batteryFatigue: 0.05,
      incentiveClass: "premium",
      technicalReadiness: true,
      locationPriority: 0.8,
      fleetBonus: 0.0
    },
    surplusKWh: 72.5,
    maxDischargeKW: 90,
    lastRotation: new Date(Date.now() - 72 * 60 * 60 * 1000).toISOString(),
    weeklyDischarges: 1
  }
];

// Mock hot zones data
const mockHotZones: HotZone[] = [
  {
    id: "zone-central",
    name: "Central Business District",
    bounds: {
      north: 1.3650,
      south: 1.3450,
      east: 103.8350,
      west: 103.8150
    },
    stressLevel: 9,
    active: true,
    priority: 25
  },
  {
    id: "zone-industrial",
    name: "Industrial Park East",
    bounds: {
      north: 1.3780,
      south: 1.3650,
      east: 103.8450,
      west: 103.8250
    },
    stressLevel: 7,
    active: true,
    priority: 15
  },
  {
    id: "zone-residential",
    name: "Residential North",
    bounds: {
      north: 1.3850,
      south: 1.3700,
      east: 103.8200,
      west: 103.8000
    },
    stressLevel: 5,
    active: false,
    priority: 10
  }
];

// TODO: remove mock functionality - V2G aggregation calls
const mockV2GCalls: V2GCall[] = [
  {
    id: "call-1",
    lat: 1.3521,
    lng: 103.8198,
    timestamp: new Date(Date.now() - 2 * 60000).toISOString(), // 2 minutes ago
    magnitude: 8,
    region: "Central-A"
  },
  {
    id: "call-2", 
    lat: 1.3621,
    lng: 103.8298,
    timestamp: new Date(Date.now() - 8 * 60000).toISOString(), // 8 minutes ago
    magnitude: 6,
    region: "East-B"
  },
  {
    id: "call-3",
    lat: 1.3721,
    lng: 103.8098,
    timestamp: new Date(Date.now() - 15 * 60000).toISOString(), // 15 minutes ago
    magnitude: 9,
    region: "North-C"
  },
  {
    id: "call-4",
    lat: 1.3421,
    lng: 103.8398,
    timestamp: new Date(Date.now() - 30 * 60000).toISOString(), // 30 minutes ago
    magnitude: 7,
    region: "South-D"
  },
  {
    id: "call-5",
    lat: 1.3321,
    lng: 103.8498,
    timestamp: new Date(Date.now() - 45 * 60000).toISOString(), // 45 minutes ago
    magnitude: 5,
    region: "West-E"
  }
];

// TODO: remove mock functionality - Grid connection invitations (peak demands)
const mockGridInvitations: GridInvitation[] = [
  {
    id: "invite-1",
    lat: 1.3545,
    lng: 103.8225,
    timestamp: new Date(Date.now() - 5 * 60000).toISOString(), // 5 minutes ago
    demandLevel: "peak",
    incentiveAmount: 15.50,
    priority: 10,
    region: "Central-Peak",
    expiresAt: new Date(Date.now() + 25 * 60000).toISOString() // expires in 25 minutes
  },
  {
    id: "invite-2",
    lat: 1.3645,
    lng: 103.8325,
    timestamp: new Date(Date.now() - 10 * 60000).toISOString(), // 10 minutes ago
    demandLevel: "high",
    incentiveAmount: 8.25,
    priority: 8,
    region: "East-High",
    expiresAt: new Date(Date.now() + 50 * 60000).toISOString() // expires in 50 minutes
  },
  {
    id: "invite-3",
    lat: 1.3745,
    lng: 103.8125,
    timestamp: new Date(Date.now() - 15 * 60000).toISOString(), // 15 minutes ago
    demandLevel: "high",
    incentiveAmount: 9.75,
    priority: 7,
    region: "North-High",
    expiresAt: new Date(Date.now() + 35 * 60000).toISOString() // expires in 35 minutes
  },
  {
    id: "invite-4",
    lat: 1.3445,
    lng: 103.8425,
    timestamp: new Date(Date.now() - 20 * 60000).toISOString(), // 20 minutes ago
    demandLevel: "medium",
    incentiveAmount: 4.50,
    priority: 5,
    region: "South-Medium",
    expiresAt: new Date(Date.now() + 70 * 60000).toISOString() // expires in 70 minutes
  },
  {
    id: "invite-5",
    lat: 1.3345,
    lng: 103.8525,
    timestamp: new Date(Date.now() - 30 * 60000).toISOString(), // 30 minutes ago
    demandLevel: "peak",
    incentiveAmount: 18.75,
    priority: 9,
    region: "West-Peak",
    expiresAt: new Date(Date.now() + 15 * 60000).toISOString() // expires in 15 minutes
  }
];

type DashboardPage = "overview" | "grid-requirements" | "vehicles" | "analytics" | "criteria" | "incentives" | "asset-management" | "settings";

function V2GPlatformContent() {
  const [vehicles, setVehicles] = useState<Vehicle[]>(mockVehicles);
  const [selectedVehicle, setSelectedVehicle] = useState<Vehicle | null>(null);
  const [selectedRegion, setSelectedRegion] = useState<RegionStat | null>(null);
  const [v2gCalls, setV2GCalls] = useState<V2GCall[]>(mockV2GCalls);
  const [gridInvitations, setGridInvitations] = useState<GridInvitation[]>(mockGridInvitations);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [activePage, setActivePage] = useState<DashboardPage>("overview");
  const [selectedHotZone, setSelectedHotZone] = useState<HotZone | null>(null);
  const isMobile = useIsMobile();
  const { pings, unreadCount, subscribe, dismissPing } = useNotifications();
  
  // Vehicle filter states for map interactivity
  const [showChargingVehicles, setShowChargingVehicles] = useState(true);
  const [showDischargingVehicles, setShowDischargingVehicles] = useState(true);
  const [showIdleVehicles, setShowIdleVehicles] = useState(true);
  const [showOfflineVehicles, setShowOfflineVehicles] = useState(true);
  const [showSocInfo, setShowSocInfo] = useState(true);
  
  // Mobile filter state
  const [filterState, setFilterState] = useState({
    showRegions: true,
    showV2GCalls: true,
    showGridInvitations: true,
    showVehicles: true,
    showScoreBubbles: true,
    showHotZones: true,
    showCharging: showChargingVehicles,
    showDischarging: showDischargingVehicles,
    showIdle: showIdleVehicles,
    showOffline: showOfflineVehicles,
    showSocInfo,
    socRange: [0, 100] as [number, number],
    batteryRange: [10, 150] as [number, number],
  });

  const handleVehicleSelect = (vehicle: Vehicle) => {
    console.log(`Selected vehicle: ${vehicle.name}`);
    setSelectedVehicle(vehicle);
    setSelectedRegion(null); // Clear region selection when vehicle is selected
  };

  const handleRegionSelect = (region: RegionStat) => {
    console.log(`Selected region: ${region.regionId} with ${region.chargingCount} charging vehicles`);
    setSelectedRegion(region);
    setSelectedVehicle(null); // Clear vehicle selection when region is selected
  };

  const handleDispatchSignal = (vehicleId: number) => {
    console.log(`Dispatching signal to vehicle ${vehicleId}`);
    // TODO: Implement actual dispatch logic
    setVehicles(prev => prev.map(v => 
      v.id === vehicleId 
        ? { ...v, status: v.status === "Idle" ? "Discharging" : "Idle" as any }
        : v
    ));
  };

  const handleRefresh = async () => {
    setIsRefreshing(true);
    console.log("Refreshing vehicle, V2G calls, and grid invitation data...");
    // TODO: Implement actual data refresh
    
    // Simulate new V2G call
    const newCall: V2GCall = {
      id: `call-${Date.now()}`,
      lat: 1.3500 + (Math.random() - 0.5) * 0.1,
      lng: 103.8300 + (Math.random() - 0.5) * 0.1,
      timestamp: new Date().toISOString(),
      magnitude: Math.floor(Math.random() * 5) + 5,
      region: `Region-${Math.floor(Math.random() * 10)}`
    };
    
    // Simulate new grid invitation (peak demand)
    const demandLevels = ['peak', 'high', 'medium'];
    const randomDemand = demandLevels[Math.floor(Math.random() * demandLevels.length)];
    const newInvitation: GridInvitation = {
      id: `invite-${Date.now()}`,
      lat: 1.3500 + (Math.random() - 0.5) * 0.1,
      lng: 103.8300 + (Math.random() - 0.5) * 0.1,
      timestamp: new Date().toISOString(),
      demandLevel: randomDemand,
      incentiveAmount: randomDemand === 'peak' ? 12 + Math.random() * 8 : 
                      randomDemand === 'high' ? 6 + Math.random() * 6 : 
                      2 + Math.random() * 4,
      priority: randomDemand === 'peak' ? 8 + Math.floor(Math.random() * 3) :
                randomDemand === 'high' ? 6 + Math.floor(Math.random() * 3) :
                3 + Math.floor(Math.random() * 4),
      region: `${randomDemand.toUpperCase()}-${Math.floor(Math.random() * 10)}`,
      expiresAt: new Date(Date.now() + (30 + Math.random() * 90) * 60000).toISOString() // 30-120 minutes
    };
    
    setV2GCalls(prev => [newCall, ...prev.slice(0, 9)]); // Keep last 10 calls
    setGridInvitations(prev => [newInvitation, ...prev.slice(0, 9)]); // Keep last 10 invitations
    
    setTimeout(() => {
      setIsRefreshing(false);
    }, 2000);
  };

  const getStatusCounts = () => {
    return vehicles.reduce((acc, vehicle) => {
      acc[vehicle.status.toLowerCase()] = (acc[vehicle.status.toLowerCase()] || 0) + 1;
      return acc;
    }, {} as Record<string, number>);
  };

  // Filter vehicles based on status toggles
  const filteredVehicles = vehicles.filter(vehicle => {
    switch (vehicle.status) {
      case "Charging":
        return showChargingVehicles;
      case "Discharging":
        return showDischargingVehicles;
      case "Idle":
        return showIdleVehicles;
      case "Offline":
        return showOfflineVehicles;
      default:
        return true;
    }
  });

  const statusCounts = getStatusCounts();
  const totalCapacity = vehicles.reduce((sum, v) => sum + v.battery, 0);
  const averageSOC = Math.round(vehicles.reduce((sum, v) => sum + v.soc, 0) / vehicles.length);

  // Mobile Layout
  if (isMobile) {
    const mobilePages = {
      "/": "overview",
      "/grid-requirements": "grid-requirements",
      "/criteria": "criteria",
      "/incentive-simulation": "incentives",
      "/settings": "settings"
    } as const;

    // Map current URL to activePage for mobile navigation
    const [location] = useLocation();
    const currentMobilePage = mobilePages[location as keyof typeof mobilePages] || "overview";
    
    // Render mobile content based on active page
    const renderMobileContent = () => {
      if (currentMobilePage === "settings") {
        return <Settings />;
      } else if (currentMobilePage === "grid-requirements") {
        return <GridRequirements />;
      } else if (currentMobilePage === "criteria") {
        return <CriteriaManagement />;
      } else if (currentMobilePage === "incentives") {
        return <IncentiveSimulation />;
      } else {
        // Default to overview/map for mobile
        return (
          <div className="flex-1 h-full">
            <VehicleMap
              vehicles={filteredVehicles}
              selectedVehicle={selectedVehicle}
              onVehicleSelect={handleVehicleSelect}
              v2gCalls={v2gCalls}
              gridInvitations={gridInvitations}
              vehicleScores={mockVehicleScores}
              hotZones={mockHotZones}
              onRegionSelect={handleRegionSelect}
              onHotZoneSelect={setSelectedHotZone}
              className="h-full w-full"
              showScoreBubbles={true}
              showHotZones={true}
              allowHotZoneEditing={true}
              showSocInfo={showSocInfo}
            />
          </div>
        );
      }
    };

    return (
      <MobileShell 
        unreadPings={unreadCount}
        showNotificationBanner={pings.length > 0}
      >
        {renderMobileContent()}
      </MobileShell>
    );
  }

  // Desktop Layout with Sidebar
  const style = {
    "--sidebar-width": "16rem",
    "--sidebar-width-icon": "3rem",
  };

  return (
    <SidebarProvider style={style as React.CSSProperties}>
      <div className="flex h-screen w-full">
        <AppSidebar activePage={activePage} onPageChange={setActivePage} />
        
        <div className="flex flex-col flex-1 overflow-hidden">
          {/* Header */}
          <header className="border-b bg-card/50 backdrop-blur-sm sticky top-0 z-50">
            <div className="px-4 py-3">
              <div className="flex items-center justify-between gap-4">
                <div className="flex items-center gap-3">
                  <SidebarTrigger data-testid="button-sidebar-toggle" />
                  <motion.div 
                    className="flex items-center gap-2"
                    initial={{ opacity: 0, x: -20 }}
                    animate={{ opacity: 1, x: 0 }}
                  >
                    <Zap className="w-6 h-6 text-primary" />
                    <div>
                      <h1 className="text-xl font-bold" data-testid="title-platform">
                        V2G Grid Management
                      </h1>
                    </div>
                  </motion.div>
                </div>

                <div className="flex items-center gap-2">
                  <Button 
                    variant="outline" 
                    size="sm" 
                    onClick={handleRefresh}
                    disabled={isRefreshing}
                    data-testid="button-refresh"
                  >
                    <RefreshCw className={`w-4 h-4 mr-2 ${isRefreshing ? 'animate-spin' : ''}`} />
                    Refresh
                  </Button>
                  <ThemeToggle />
                </div>
              </div>
            </div>
          </header>

          {/* Main Content Area */}
          <div className="flex-1 overflow-auto">
            {/* Action Buttons Bar */}
            {activePage === "overview" && (
        <div className="px-6 py-4 border-b bg-card/30">
          <div className="flex flex-wrap gap-2">
            <Button
              variant={showChargingVehicles ? "default" : "outline"}
              size="sm"
              onClick={() => setShowChargingVehicles(!showChargingVehicles)}
              data-testid="filter-charging"
            >
              <Battery className="w-4 h-4 mr-2" />
              Charging
              {showChargingVehicles ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>

            <Button
              variant={showDischargingVehicles ? "default" : "outline"}
              size="sm"
              onClick={() => setShowDischargingVehicles(!showDischargingVehicles)}
              data-testid="filter-discharging"
            >
              <Zap className="w-4 h-4 mr-2" />
              Discharging
              {showDischargingVehicles ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>

            <Button
              variant={showIdleVehicles ? "default" : "outline"}
              size="sm"
              onClick={() => setShowIdleVehicles(!showIdleVehicles)}
              data-testid="filter-idle"
            >
              <Car className="w-4 h-4 mr-2" />
              Idle
              {showIdleVehicles ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>

            <Button
              variant={showOfflineVehicles ? "default" : "outline"}
              size="sm"
              onClick={() => setShowOfflineVehicles(!showOfflineVehicles)}
              data-testid="filter-offline"
            >
              <AlertCircle className="w-4 h-4 mr-2" />
              Offline
              {showOfflineVehicles ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>

            <Button
              variant={showSocInfo ? "default" : "outline"}
              size="sm"
              onClick={() => setShowSocInfo(!showSocInfo)}
              data-testid="filter-soc-info"
            >
              <TrendingUp className="w-4 h-4 mr-2" />
              SOC Info
              {showSocInfo ? <Eye className="w-3 h-3 ml-2" /> : <EyeOff className="w-3 h-3 ml-2" />}
            </Button>
          </div>
        </div>
      )}

        {/* Overview Page */}
        {activePage === "overview" && (
          <div className="p-6">
        <div className="grid grid-cols-1 lg:grid-cols-3 gap-6 h-[calc(100vh-280px)]">
          {/* Map Section */}
          <motion.div 
            className="lg:col-span-2"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.1 }}
          >
            <VehicleMap 
              vehicles={filteredVehicles}
              selectedVehicle={selectedVehicle}
              onVehicleSelect={handleVehicleSelect}
              v2gCalls={v2gCalls}
              gridInvitations={gridInvitations}
              vehicleScores={mockVehicleScores}
              hotZones={mockHotZones}
              onRegionSelect={handleRegionSelect}
              onHotZoneSelect={setSelectedHotZone}
              className="h-full"
              showScoreBubbles={true}
              showHotZones={true}
              allowHotZoneEditing={true}
              showSocInfo={showSocInfo}
            />
          </motion.div>

          {/* Side Panel */}
          <motion.div 
            className="space-y-6"
            initial={{ opacity: 0, y: 20 }}
            animate={{ opacity: 1, y: 0 }}
            transition={{ delay: 0.2 }}
          >
            {/* Vehicle/Region Info Panel */}
            <div className="h-[400px]">
              {selectedRegion ? (
                <Card className="h-full">
                  <CardHeader>
                    <CardTitle className="flex items-center gap-2">
                      <div className="w-3 h-3 bg-primary rounded"></div>
                      Region {selectedRegion.regionId}
                    </CardTitle>
                  </CardHeader>
                  <CardContent className="space-y-4" data-testid="region-info-panel">
                    <div className="grid grid-cols-2 gap-4">
                      <div className="space-y-2">
                        <div className="text-sm text-muted-foreground">Total Vehicles</div>
                        <div className="text-2xl font-bold" data-testid="region-total-vehicles">
                          {selectedRegion.vehicleCount}
                        </div>
                      </div>
                      <div className="space-y-2">
                        <div className="text-sm text-muted-foreground">Avg SOC</div>
                        <div className="text-2xl font-bold" data-testid="region-avg-soc">
                          {selectedRegion.averageSOC}%
                        </div>
                      </div>
                    </div>
                    
                    <div className="space-y-2">
                      <div className="text-sm font-medium">Vehicle Status Breakdown</div>
                      <div className="space-y-1">
                        <div className="flex justify-between items-center">
                          <div className="flex items-center gap-2">
                            <div className="w-2 h-2 bg-vehicle-charging rounded-full"></div>
                            <span className="text-sm">Charging</span>
                          </div>
                          <span className="font-medium" data-testid="region-charging-count">
                            {selectedRegion.chargingCount}
                          </span>
                        </div>
                        <div className="flex justify-between items-center">
                          <div className="flex items-center gap-2">
                            <div className="w-2 h-2 bg-vehicle-discharging rounded-full"></div>
                            <span className="text-sm">Discharging</span>
                          </div>
                          <span className="font-medium">
                            {selectedRegion.dischargingCount}
                          </span>
                        </div>
                        <div className="flex justify-between items-center">
                          <div className="flex items-center gap-2">
                            <div className="w-2 h-2 bg-vehicle-idle rounded-full"></div>
                            <span className="text-sm">Idle</span>
                          </div>
                          <span className="font-medium">
                            {selectedRegion.idleCount}
                          </span>
                        </div>
                        <div className="flex justify-between items-center">
                          <div className="flex items-center gap-2">
                            <div className="w-2 h-2 bg-vehicle-error rounded-full"></div>
                            <span className="text-sm">Offline</span>
                          </div>
                          <span className="font-medium">
                            {selectedRegion.offlineCount}
                          </span>
                        </div>
                      </div>
                    </div>
                    
                    <div className="pt-2 border-t">
                      <div className="text-sm text-muted-foreground">Total Capacity</div>
                      <div className="text-lg font-bold">
                        {selectedRegion.totalCapacity} kWh
                      </div>
                    </div>
                  </CardContent>
                </Card>
              ) : (
                <VehicleInfoPanel 
                  selectedVehicle={selectedVehicle}
                  onDispatchSignal={handleDispatchSignal}
                  className="h-full"
                  showSocInfo={showSocInfo}
                />
              )}
            </div>

          </motion.div>
        </div>
          </div>
        )}

        {/* Grid Requirements Page */}
        {activePage === "grid-requirements" && <GridRequirements />}

        {/* Vehicles Dashboard Page */}
        {activePage === "vehicles" && <VehiclesDashboard />}

        {/* Charging Analytics Page */}
        {activePage === "analytics" && <ChargingAnalytics />}

        {/* Criteria Management Page */}
        {activePage === "criteria" && <CriteriaManagement />}
        
        {/* Incentive Simulation Page */}
        {activePage === "incentives" && <IncentiveSimulation />}
        
        {/* Asset Management Page */}
        {activePage === "asset-management" && <AssetManagement />}
        
        {/* Settings Page */}
        {activePage === "settings" && <Settings />}
          </div>
        </div>
      </div>
    </SidebarProvider>
  );
}

export default function V2GPlatform() {
  return (
    <NotificationProvider>
      <V2GPlatformContent />
    </NotificationProvider>
  );
}