import { 
  Sidebar, 
  SidebarContent, 
  SidebarGroup, 
  SidebarGroupContent, 
  SidebarGroupLabel,
  SidebarMenu, 
  SidebarMenuButton, 
  SidebarMenuItem,
  SidebarFooter,
  SidebarHeader
} from "@/components/ui/sidebar";
import { 
  LayoutDashboard, 
  Zap, 
  Car, 
  BarChart3, 
  Settings2, 
  Calculator, 
  Package,
  LogOut,
  User
} from "lucide-react";
import { useAuth } from "@/hooks/use-auth";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback } from "@/components/ui/avatar";
import { Separator } from "@/components/ui/separator";

type DashboardPage = "overview" | "grid-requirements" | "vehicles" | "analytics" | "criteria" | "incentives" | "asset-management" | "settings";

const menuItems = [
  {
    title: "Overview",
    icon: LayoutDashboard,
    page: "overview" as const,
    testId: "nav-overview",
  },
  {
    title: "Grid Requirements",
    icon: Zap,
    page: "grid-requirements" as const,
    testId: "nav-grid-requirements",
  },
  {
    title: "Vehicles",
    icon: Car,
    page: "vehicles" as const,
    testId: "nav-vehicles",
  },
  {
    title: "Analytics",
    icon: BarChart3,
    page: "analytics" as const,
    testId: "nav-analytics",
  },
  {
    title: "Criteria",
    icon: Settings2,
    page: "criteria" as const,
    testId: "nav-criteria",
  },
  {
    title: "Incentives",
    icon: Calculator,
    page: "incentives" as const,
    testId: "nav-incentives",
  },
  {
    title: "Asset Management",
    icon: Package,
    page: "asset-management" as const,
    testId: "nav-asset-management",
  },
];

export function AppSidebar({ 
  activePage, 
  onPageChange 
}: { 
  activePage: DashboardPage;
  onPageChange: (page: DashboardPage) => void;
}) {
  const { user, logoutMutation } = useAuth();

  const handleLogout = () => {
    logoutMutation.mutate();
  };

  const getUserInitials = () => {
    if (!user) return "U";
    return user.username.substring(0, 2).toUpperCase();
  };

  return (
    <Sidebar>
      <SidebarHeader className="p-4">
        <div className="flex items-center gap-2">
          <Zap className="h-6 w-6 text-primary" />
          <div>
            <h2 className="text-lg font-semibold">V2G Platform</h2>
            <p className="text-xs text-muted-foreground">Grid Management</p>
          </div>
        </div>
      </SidebarHeader>
      
      <Separator />
      
      <SidebarContent>
        <SidebarGroup>
          <SidebarGroupLabel>Navigation</SidebarGroupLabel>
          <SidebarGroupContent>
            <SidebarMenu>
              {menuItems.map((item) => (
                <SidebarMenuItem key={item.page}>
                  <SidebarMenuButton
                    onClick={() => onPageChange(item.page)}
                    isActive={activePage === item.page}
                    data-testid={item.testId}
                  >
                    <item.icon className="h-4 w-4" />
                    <span>{item.title}</span>
                  </SidebarMenuButton>
                </SidebarMenuItem>
              ))}
            </SidebarMenu>
          </SidebarGroupContent>
        </SidebarGroup>
      </SidebarContent>

      <SidebarFooter className="p-4">
        <div className="space-y-2">
          <div className="flex items-center gap-3 p-2 rounded-md bg-sidebar-accent/50">
            <Avatar className="h-8 w-8">
              <AvatarFallback className="bg-primary/10 text-primary">
                <User className="h-4 w-4" />
              </AvatarFallback>
            </Avatar>
            <div className="flex-1 min-w-0">
              <p className="text-sm font-medium truncate" data-testid="text-username">
                {user?.username}
              </p>
              <p className="text-xs text-muted-foreground">Operator</p>
            </div>
          </div>
          <Button 
            variant="outline" 
            size="sm" 
            className="w-full"
            onClick={handleLogout}
            disabled={logoutMutation.isPending}
            data-testid="button-logout"
          >
            <LogOut className="h-4 w-4 mr-2" />
            {logoutMutation.isPending ? "Logging out..." : "Logout"}
          </Button>
        </div>
      </SidebarFooter>
    </Sidebar>
  );
}
