import { useState } from "react";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { 
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import {
  Table,
  TableBody,
  TableCell,
  TableHead,
  TableHeader,
  TableRow,
} from "@/components/ui/table";
import {
  Accordion,
  AccordionContent,
  AccordionItem,
  AccordionTrigger,
} from "@/components/ui/accordion";
import { Slider } from "@/components/ui/slider";
import { Switch } from "@/components/ui/switch";
import {
  Sheet,
  SheetContent,
  SheetDescription,
  SheetHeader,
  SheetTitle,
} from "@/components/ui/sheet";
import { Checkbox } from "@/components/ui/checkbox";
import { 
  Package, 
  Car, 
  Battery, 
  MapPin,
  Calendar,
  TrendingUp,
  AlertCircle,
  CheckCircle2,
  Clock,
  Settings,
  Building2,
  Activity,
  Radio,
  DollarSign,
  Gift,
  Flame,
  Bell,
  Download,
  RefreshCw,
  Lock,
  Unlock,
  Zap,
  FileText,
  Phone,
  Target,
  BarChart3,
  Users,
  Send
} from "lucide-react";
import { 
  LineChart, 
  Line, 
  XAxis, 
  YAxis, 
  CartesianGrid, 
  Tooltip, 
  ResponsiveContainer,
  Area,
  AreaChart
} from "recharts";

// Chargers Tab Component
function ChargersTab() {
  const [searchQuery, setSearchQuery] = useState("");
  const [selectedChargers, setSelectedChargers] = useState<string[]>([]);
  const [detailDrawerOpen, setDetailDrawerOpen] = useState(false);
  const [selectedChargerDetail, setSelectedChargerDetail] = useState<any>(null);
  const [detailTab, setDetailTab] = useState("overview");

  const mockChargers = [
    { id: "CHG-001", site: "Marina Bay", model: "Tesla Supercharger V3", firmware: "2.5.3", power: 250, status: "active", lastHeartbeat: "2min ago", alerts: 0, sessions24h: 45, v2gEnabled: true, connectors: 2 },
    { id: "CHG-002", site: "Orchard Central", model: "ABB Terra 360", firmware: "2.5.1", power: 360, status: "active", lastHeartbeat: "1min ago", alerts: 1, sessions24h: 67, v2gEnabled: true, connectors: 1 },
    { id: "CHG-003", site: "Changi Airport", model: "Tesla Supercharger V3", firmware: "2.5.3", power: 250, status: "maintenance", lastHeartbeat: "15min ago", alerts: 3, sessions24h: 12, v2gEnabled: false, connectors: 2 },
    { id: "CHG-004", site: "Jurong East", model: "ChargePoint CP6000", firmware: "3.1.2", power: 150, status: "active", lastHeartbeat: "3min ago", alerts: 0, sessions24h: 38, v2gEnabled: true, connectors: 2 },
  ];

  const filteredChargers = mockChargers.filter(c => 
    c.id.toLowerCase().includes(searchQuery.toLowerCase()) ||
    c.site.toLowerCase().includes(searchQuery.toLowerCase())
  );

  const handleCheckboxChange = (chargerId: string, checked: boolean) => {
    if (checked) {
      setSelectedChargers([...selectedChargers, chargerId]);
    } else {
      setSelectedChargers(selectedChargers.filter(id => id !== chargerId));
    }
  };

  const handleViewCharger = (charger: any) => {
    setSelectedChargerDetail(charger);
    setDetailDrawerOpen(true);
    setDetailTab("overview");
  };

  const mockSessions = [
    { id: "SES-001", vehicleId: "EV-101", userId: "user-001", manufacturer: "Tesla", time: "2hr ago", kWh: 50, outcome: "Complete", earnings: "$25.00", status: "completed" },
    { id: "SES-002", vehicleId: "EV-202", userId: "user-002", manufacturer: "BMW", time: "5hr ago", kWh: 50, outcome: "Complete", earnings: "$25.00", status: "completed" },
    { id: "SES-003", vehicleId: "EV-101", userId: "user-001", manufacturer: "Tesla", time: "1day ago", kWh: 50, outcome: "Interrupted", earnings: "$20.00", status: "cancelled" },
  ];

  const mockAlerts = [
    { id: "ALT-001", severity: "warning", message: "Temperature threshold exceeded", timestamp: "15min ago", status: "Open" },
    { id: "ALT-002", severity: "info", message: "Firmware update available", timestamp: "2hr ago", status: "Closed" },
  ];

  return (
    <div className="space-y-4">
      {/* Top Bar */}
      <div className="flex gap-4 items-center">
        <Input
          placeholder="Quick search chargers..."
          value={searchQuery}
          onChange={(e) => setSearchQuery(e.target.value)}
          className="flex-1"
          data-testid="input-search-chargers"
        />
        <div className="flex gap-2">
          <Button variant="outline" size="sm" disabled={selectedChargers.length === 0} data-testid="button-bulk-reboot">
            <RefreshCw className="w-4 h-4 mr-2" />
            Reboot ({selectedChargers.length})
          </Button>
          <Button variant="outline" size="sm" disabled={selectedChargers.length === 0} data-testid="button-bulk-lock">
            <Lock className="w-4 h-4 mr-2" />
            Lock ({selectedChargers.length})
          </Button>
          <Button variant="outline" size="sm" disabled={selectedChargers.length === 0} data-testid="button-bulk-cap">
            <Zap className="w-4 h-4 mr-2" />
            Set Cap ({selectedChargers.length})
          </Button>
        </div>
      </div>

      {/* Data Grid */}
      <Card>
        <CardContent className="p-0">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead className="w-12">Select</TableHead>
                <TableHead>Asset ID</TableHead>
                <TableHead>Site</TableHead>
                <TableHead>Model</TableHead>
                <TableHead>Firmware</TableHead>
                <TableHead>Power (kW)</TableHead>
                <TableHead>Status</TableHead>
                <TableHead>Last Heartbeat</TableHead>
                <TableHead>Alerts</TableHead>
                <TableHead>Sessions (24h)</TableHead>
                <TableHead>Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {filteredChargers.map((charger) => (
                <TableRow key={charger.id} data-testid={`row-charger-${charger.id}`}>
                  <TableCell>
                    <Checkbox
                      checked={selectedChargers.includes(charger.id)}
                      onCheckedChange={(checked) => handleCheckboxChange(charger.id, !!checked)}
                      data-testid={`checkbox-charger-${charger.id}`}
                    />
                  </TableCell>
                  <TableCell className="font-medium">{charger.id}</TableCell>
                  <TableCell>{charger.site}</TableCell>
                  <TableCell>{charger.model}</TableCell>
                  <TableCell>{charger.firmware}</TableCell>
                  <TableCell>{charger.power} kW</TableCell>
                  <TableCell>
                    <Badge variant={charger.status === "active" ? "default" : "secondary"}>
                      {charger.status}
                    </Badge>
                  </TableCell>
                  <TableCell>{charger.lastHeartbeat}</TableCell>
                  <TableCell>
                    {charger.alerts > 0 ? (
                      <Badge variant="destructive">{charger.alerts}</Badge>
                    ) : (
                      <span className="text-muted-foreground">0</span>
                    )}
                  </TableCell>
                  <TableCell>{charger.sessions24h}</TableCell>
                  <TableCell>
                    <Button 
                      variant="ghost" 
                      size="sm" 
                      onClick={() => handleViewCharger(charger)}
                      data-testid={`button-view-${charger.id}`}
                    >
                      View
                    </Button>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      {/* Detail Drawer */}
      <Sheet open={detailDrawerOpen} onOpenChange={setDetailDrawerOpen}>
        <SheetContent className="w-full sm:max-w-2xl overflow-y-auto">
          <SheetHeader>
            <SheetTitle>Charger Details: {selectedChargerDetail?.id}</SheetTitle>
            <SheetDescription>{selectedChargerDetail?.model} at {selectedChargerDetail?.site}</SheetDescription>
          </SheetHeader>

          <Tabs value={detailTab} onValueChange={setDetailTab} className="mt-6">
            <TabsList className="grid grid-cols-5 w-full">
              <TabsTrigger value="overview" data-testid="detail-tab-overview">Overview</TabsTrigger>
              <TabsTrigger value="sessions" data-testid="detail-tab-sessions">Sessions</TabsTrigger>
              <TabsTrigger value="alerts" data-testid="detail-tab-alerts">Alerts</TabsTrigger>
              <TabsTrigger value="firmware" data-testid="detail-tab-firmware">Firmware</TabsTrigger>
              <TabsTrigger value="install" data-testid="detail-tab-install">Install</TabsTrigger>
            </TabsList>

            <TabsContent value="overview" className="space-y-4 mt-4">
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Device Information</CardTitle>
                </CardHeader>
                <CardContent className="space-y-3">
                  <div className="grid grid-cols-2 gap-4">
                    <div>
                      <p className="text-sm text-muted-foreground">Device Name</p>
                      <p className="font-medium">{selectedChargerDetail?.model}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Asset ID</p>
                      <p className="font-medium">{selectedChargerDetail?.id}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Site Location</p>
                      <p className="font-medium">{selectedChargerDetail?.site}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">V2G Enabled</p>
                      <Badge variant={selectedChargerDetail?.v2gEnabled ? "default" : "secondary"}>
                        {selectedChargerDetail?.v2gEnabled ? "Yes" : "No"}
                      </Badge>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Connectors</p>
                      <p className="font-medium">{selectedChargerDetail?.connectors}</p>
                    </div>
                    <div>
                      <p className="text-sm text-muted-foreground">Power Rating</p>
                      <p className="font-medium">{selectedChargerDetail?.power} kW</p>
                    </div>
                  </div>

                  <div className="pt-4 space-y-2">
                    <p className="text-sm font-medium">Health Status</p>
                    <div className="grid grid-cols-2 gap-2">
                      <div className="flex items-center gap-2">
                        <CheckCircle2 className="w-4 h-4 text-green-500" />
                        <span className="text-sm">Heartbeat: Active</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <CheckCircle2 className="w-4 h-4 text-green-500" />
                        <span className="text-sm">OCPP: Connected</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <CheckCircle2 className="w-4 h-4 text-green-500" />
                        <span className="text-sm">Signal: Strong</span>
                      </div>
                      <div className="flex items-center gap-2">
                        <CheckCircle2 className="w-4 h-4 text-green-500" />
                        <span className="text-sm">Temperature: Normal</span>
                      </div>
                    </div>
                  </div>
                </CardContent>
              </Card>

              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Controls</CardTitle>
                </CardHeader>
                <CardContent>
                  <div className="flex flex-wrap gap-2">
                    <Button variant="outline" size="sm">
                      <RefreshCw className="w-4 h-4 mr-2" />
                      Reboot
                    </Button>
                    <Button variant="outline" size="sm">
                      <Lock className="w-4 h-4 mr-2" />
                      Lock
                    </Button>
                    <Button variant="outline" size="sm">
                      <Zap className="w-4 h-4 mr-2" />
                      Set Power Cap
                    </Button>
                    <Button variant="outline" size="sm">
                      <AlertCircle className="w-4 h-4 mr-2" />
                      Safe Shutdown
                    </Button>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="sessions" className="mt-4">
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Recent Sessions</CardTitle>
                </CardHeader>
                <CardContent className="p-0">
                  <Table>
                    <TableHeader>
                      <TableRow>
                        <TableHead>Session ID</TableHead>
                        <TableHead>Time</TableHead>
                        <TableHead>Energy (kWh)</TableHead>
                        <TableHead>Outcome</TableHead>
                        <TableHead>Earnings</TableHead>
                      </TableRow>
                    </TableHeader>
                    <TableBody>
                      {mockSessions.map((session) => (
                        <TableRow key={session.id}>
                          <TableCell className="font-medium">{session.id}</TableCell>
                          <TableCell>{session.time}</TableCell>
                          <TableCell>{session.kWh} kWh</TableCell>
                          <TableCell>
                            <Badge variant={session.outcome === "Complete" ? "default" : "secondary"}>
                              {session.outcome}
                            </Badge>
                          </TableCell>
                          <TableCell>{session.earnings}</TableCell>
                        </TableRow>
                      ))}
                    </TableBody>
                  </Table>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="alerts" className="mt-4">
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Alerts & Logs</CardTitle>
                </CardHeader>
                <CardContent className="space-y-3">
                  {mockAlerts.map((alert) => (
                    <div key={alert.id} className="flex items-start justify-between p-3 border rounded-lg">
                      <div className="flex-1">
                        <div className="flex items-center gap-2 mb-1">
                          <Badge variant={alert.severity === "warning" ? "destructive" : "secondary"}>
                            {alert.severity}
                          </Badge>
                          <span className="text-sm font-medium">{alert.message}</span>
                        </div>
                        <p className="text-sm text-muted-foreground">{alert.timestamp}</p>
                      </div>
                      <Badge variant="outline">{alert.status}</Badge>
                    </div>
                  ))}
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="firmware" className="mt-4">
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Firmware Management</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <p className="text-sm text-muted-foreground">Current Version</p>
                    <p className="text-lg font-semibold">{selectedChargerDetail?.firmware}</p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">Latest Available</p>
                    <p className="text-lg font-semibold">2.6.0</p>
                  </div>
                  <div className="space-y-2">
                    <p className="text-sm font-medium">Rollout Progress</p>
                    <div className="w-full h-2 bg-secondary rounded-full">
                      <div className="h-full bg-primary rounded-full" style={{ width: "65%" }} />
                    </div>
                    <p className="text-sm text-muted-foreground">65% of fleet updated</p>
                  </div>
                  <div className="flex gap-2">
                    <Button variant="outline" size="sm">Update Firmware</Button>
                    <Button variant="outline" size="sm">Rollback</Button>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>

            <TabsContent value="install" className="mt-4">
              <Card>
                <CardHeader>
                  <CardTitle className="text-base">Installation & Warranty</CardTitle>
                </CardHeader>
                <CardContent className="space-y-4">
                  <div>
                    <p className="text-sm text-muted-foreground">Installer</p>
                    <p className="font-medium">Singapore EV Solutions Pte Ltd</p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">Installation Date</p>
                    <p className="font-medium">2023-08-15</p>
                  </div>
                  <div>
                    <p className="text-sm text-muted-foreground">Warranty Expires</p>
                    <p className="font-medium">2026-08-15</p>
                  </div>
                  <div>
                    <p className="text-sm font-medium mb-2">Installation Documents</p>
                    <div className="space-y-2">
                      <Button variant="outline" size="sm" className="w-full justify-start">
                        <FileText className="w-4 h-4 mr-2" />
                        Installation Certificate.pdf
                      </Button>
                      <Button variant="outline" size="sm" className="w-full justify-start">
                        <FileText className="w-4 h-4 mr-2" />
                        Warranty Document.pdf
                      </Button>
                    </div>
                  </div>
                </CardContent>
              </Card>
            </TabsContent>
          </Tabs>
        </SheetContent>
      </Sheet>
    </div>
  );
}

// Sites Tab Component
function SitesTab() {
  const mockSites = [
    { id: "SITE-001", name: "Marina Bay Hub", capacity: 500, current: 380, chargers: 8, tariff: "Premium", limit: 500 },
    { id: "SITE-002", name: "Orchard Central", capacity: 300, current: 245, chargers: 5, tariff: "Standard", limit: 300 },
    { id: "SITE-003", name: "Changi Airport", capacity: 800, current: 620, chargers: 12, tariff: "Premium", limit: 800 },
  ];

  const loadChartData = Array.from({ length: 24 }, (_, i) => ({
    hour: `${i}:00`,
    load: Math.random() * 400 + 100,
    limit: 500,
    forecast: Math.random() * 350 + 150
  }));

  return (
    <div className="space-y-4">
      {mockSites.map((site) => (
        <Card key={site.id} data-testid={`card-site-${site.id}`}>
          <CardHeader>
            <CardTitle className="flex items-center justify-between">
              <div className="flex items-center gap-2">
                <Building2 className="w-5 h-5" />
                {site.name}
              </div>
              <div className="flex gap-2">
                <Badge>{site.tariff}</Badge>
                <Badge variant="outline">{site.chargers} Chargers</Badge>
              </div>
            </CardTitle>
          </CardHeader>
          <CardContent className="space-y-4">
            {/* Capacity Bar */}
            <div className="space-y-2">
              <div className="flex justify-between text-sm">
                <span className="text-muted-foreground">Capacity Usage</span>
                <span className="font-medium">{site.current} / {site.capacity} kW</span>
              </div>
              <div className="w-full h-3 bg-secondary rounded-full overflow-hidden">
                <div 
                  className="h-full bg-primary transition-all"
                  style={{ width: `${(site.current / site.capacity) * 100}%` }}
                />
              </div>
            </div>

            {/* Load Chart */}
            <div className="h-48">
              <ResponsiveContainer width="100%" height="100%">
                <AreaChart data={loadChartData}>
                  <CartesianGrid strokeDasharray="3 3" />
                  <XAxis dataKey="hour" />
                  <YAxis />
                  <Tooltip />
                  <Area type="monotone" dataKey="limit" stroke="#888" fill="#f0f0f0" />
                  <Area type="monotone" dataKey="load" stroke="#8884d8" fill="#8884d8" fillOpacity={0.6} />
                  <Line type="monotone" dataKey="forecast" stroke="#82ca9d" strokeDasharray="5 5" />
                </AreaChart>
              </ResponsiveContainer>
            </div>

            {/* Policies */}
            <Accordion type="single" collapsible>
              <AccordionItem value="policies">
                <AccordionTrigger>Site Policies</AccordionTrigger>
                <AccordionContent className="space-y-4">
                  <div className="space-y-2">
                    <Label>Current Cap (kW)</Label>
                    <Slider defaultValue={[site.limit]} max={1000} step={10} />
                  </div>
                  <div className="flex items-center justify-between">
                    <Label>Emergency Curtail</Label>
                    <Switch />
                  </div>
                </AccordionContent>
              </AccordionItem>
              <AccordionItem value="contacts">
                <AccordionTrigger>Emergency Contacts</AccordionTrigger>
                <AccordionContent className="space-y-2">
                  <div className="flex items-center justify-between">
                    <span className="text-sm">Facilities Manager</span>
                    <Button variant="outline" size="sm">
                      <Phone className="w-4 h-4 mr-2" />
                      Call
                    </Button>
                  </div>
                </AccordionContent>
              </AccordionItem>
            </Accordion>
          </CardContent>
        </Card>
      ))}
    </div>
  );
}

// Vehicles (EV Registry) Tab Component
function VehiclesTab() {
  const mockVehicles = [
    { 
      id: "EV-101", 
      name: "Tesla Model S",
      manufacturer: "Tesla",
      userId: "user-001",
      userName: "Sarah Chen",
      battery: 100, 
      soc: 82,
      socFloor: 20,
      maxDischargeRate: 75,
      chargeCycles: 12,
      v2gEnrolled: true,
      fleet: false,
      status: "discharging",
      lat: 1.3521,
      lng: 103.8198,
      address: "Marina Bay Charging Hub",
      district: "Central Business District",
      lastSeen: "2 min ago" 
    },
    { 
      id: "EV-202", 
      name: "BMW i4",
      manufacturer: "BMW",
      userId: "user-002",
      userName: "Marcus Wong",
      battery: 84, 
      soc: 78,
      socFloor: 30,
      maxDischargeRate: 75,
      chargeCycles: 6,
      v2gEnrolled: true,
      fleet: false,
      status: "charging",
      lat: 1.3048,
      lng: 103.8318,
      address: "Orchard Plaza Station",
      district: "Orchard Shopping District",
      lastSeen: "5 min ago" 
    },
    { 
      id: "EV-303", 
      name: "Nissan Leaf",
      manufacturer: "Nissan",
      userId: "user-003",
      userName: "Lisa Tan",
      battery: 62, 
      soc: 81,
      socFloor: 25,
      maxDischargeRate: 75,
      chargeCycles: 3,
      v2gEnrolled: false,
      fleet: false,
      status: "idle",
      lat: 1.3294,
      lng: 103.7414,
      address: "Jurong Point Mall",
      district: "Jurong Industrial Estate",
      lastSeen: "1 min ago" 
    },
    { 
      id: "EV-404", 
      name: "Volvo XC40",
      manufacturer: "Volvo",
      userId: "fleet-001",
      userName: "Fleet Services Ltd",
      battery: 78, 
      soc: 79,
      socFloor: 15,
      maxDischargeRate: 75,
      chargeCycles: 8,
      v2gEnrolled: true,
      fleet: true,
      status: "discharging",
      lat: 1.3383,
      lng: 103.8441,
      address: "Toa Payoh Hub",
      district: "Toa Payoh",
      lastSeen: "25 min ago" 
    },
    { 
      id: "EV-505", 
      name: "Audi e-tron",
      manufacturer: "Audi",
      userId: "user-005",
      userName: "David Lim",
      battery: 95, 
      soc: 68,
      socFloor: 20,
      maxDischargeRate: 75,
      chargeCycles: 15,
      v2gEnrolled: true,
      fleet: false,
      status: "idle",
      lat: 1.3200,
      lng: 103.8500,
      address: "Bugis Junction",
      district: "Bugis",
      lastSeen: "8 min ago" 
    },
    { 
      id: "EV-606", 
      name: "Mercedes EQC",
      manufacturer: "Mercedes",
      userId: "user-006",
      userName: "Rachel Koh",
      battery: 80, 
      soc: 92,
      socFloor: 25,
      maxDischargeRate: 75,
      chargeCycles: 4,
      v2gEnrolled: true,
      fleet: false,
      status: "charging",
      lat: 1.2800,
      lng: 103.8400,
      address: "Sentosa Gateway",
      district: "Sentosa",
      lastSeen: "15 min ago" 
    },
    { 
      id: "EV-707", 
      name: "BYD Atto 3",
      manufacturer: "BYD",
      userId: "user-007",
      userName: "James Ng",
      battery: 60, 
      soc: 55,
      socFloor: 30,
      maxDischargeRate: 75,
      chargeCycles: 9,
      v2gEnrolled: false,
      fleet: false,
      status: "idle",
      lat: 1.3700,
      lng: 103.8900,
      address: "Paya Lebar Quarter",
      district: "Paya Lebar",
      lastSeen: "3 min ago" 
    },
    { 
      id: "EV-808", 
      name: "Hyundai Ioniq 6",
      manufacturer: "Hyundai",
      userId: "user-008",
      userName: "Emily Tan",
      battery: 77, 
      soc: 85,
      socFloor: 20,
      maxDischargeRate: 75,
      chargeCycles: 11,
      v2gEnrolled: true,
      fleet: false,
      status: "discharging",
      lat: 1.3100,
      lng: 103.7800,
      address: "Clementi Mall",
      district: "Clementi",
      lastSeen: "6 min ago" 
    },
    { 
      id: "EV-909", 
      name: "Polestar 2",
      manufacturer: "Polestar",
      userId: "user-009",
      userName: "Kevin Lee",
      battery: 78, 
      soc: 72,
      socFloor: 25,
      maxDischargeRate: 75,
      chargeCycles: 7,
      v2gEnrolled: true,
      fleet: false,
      status: "charging",
      lat: 1.3500,
      lng: 103.9400,
      address: "Tampines Hub",
      district: "Tampines",
      lastSeen: "10 min ago" 
    },
    { 
      id: "EV-1010", 
      name: "Kia EV6",
      manufacturer: "Kia",
      userId: "fleet-002",
      userName: "Green Transport Co",
      battery: 77, 
      soc: 64,
      socFloor: 15,
      maxDischargeRate: 75,
      chargeCycles: 13,
      v2gEnrolled: true,
      fleet: true,
      status: "idle",
      lat: 1.4400,
      lng: 103.7900,
      address: "Woodlands Checkpoint",
      district: "Woodlands",
      lastSeen: "20 min ago" 
    },
  ];

  return (
    <Card>
      <CardHeader>
        <CardTitle className="flex items-center gap-2">
          <Car className="w-5 h-5" />
          EV Registry
        </CardTitle>
      </CardHeader>
      <CardContent className="p-0">
        <Table>
          <TableHeader>
            <TableRow>
              <TableHead>Vehicle ID</TableHead>
              <TableHead>User</TableHead>
              <TableHead>Make/Model</TableHead>
              <TableHead>Battery (kWh)</TableHead>
              <TableHead>SOC (%)</TableHead>
              <TableHead>V2G Enrollment</TableHead>
              <TableHead>Charge Cycles</TableHead>
              <TableHead>Location</TableHead>
              <TableHead>Status</TableHead>
              <TableHead>Actions</TableHead>
            </TableRow>
          </TableHeader>
          <TableBody>
            {mockVehicles.map((vehicle) => (
              <TableRow key={vehicle.id} data-testid={`row-vehicle-${vehicle.id}`}>
                <TableCell className="font-medium">{vehicle.id}</TableCell>
                <TableCell>{vehicle.userName}</TableCell>
                <TableCell>{vehicle.manufacturer} {vehicle.name}</TableCell>
                <TableCell>{vehicle.battery} kWh</TableCell>
                <TableCell>{vehicle.soc}%</TableCell>
                <TableCell>
                  <Badge variant={vehicle.v2gEnrolled ? "default" : "secondary"}>
                    {vehicle.v2gEnrolled ? "Enrolled" : "Not Enrolled"}
                  </Badge>
                </TableCell>
                <TableCell>{vehicle.chargeCycles}</TableCell>
                <TableCell>{vehicle.district}</TableCell>
                <TableCell>
                  <Badge variant={vehicle.status === "discharging" ? "default" : vehicle.status === "charging" ? "secondary" : "outline"}>
                    {vehicle.status}
                  </Badge>
                </TableCell>
                <TableCell>
                  <Button variant="ghost" size="sm" data-testid={`button-manage-vehicle-${vehicle.id}`}>
                    Manage
                  </Button>
                </TableCell>
              </TableRow>
            ))}
          </TableBody>
        </Table>
      </CardContent>
    </Card>
  );
}

// Health & Alerts Tab Component
function HealthAlertsTab() {
  const healthStats = {
    uptime: 99.7,
    mttr: 12,
    mtbf: 720,
    errors: { critical: 2, warning: 8, info: 15 }
  };

  const mockAlerts = [
    { id: "ALT-001", asset: "CHG-002", type: "Connectivity", severity: "warning", triggered: "15min ago", state: "Active" },
    { id: "ALT-002", asset: "CHG-003", type: "Temperature", severity: "critical", triggered: "1hr ago", state: "Acknowledged" },
    { id: "ALT-003", asset: "SITE-001", type: "Capacity", severity: "warning", triggered: "30min ago", state: "Active" },
  ];

  return (
    <div className="space-y-4">
      {/* Health Summary */}
      <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">Uptime</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-500" data-testid="stat-uptime">{healthStats.uptime}%</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">MTTR (min)</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="stat-mttr">{healthStats.mttr}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">MTBF (hrs)</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="stat-mtbf">{healthStats.mtbf}</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">Active Errors</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="flex gap-2">
              <Badge variant="destructive">{healthStats.errors.critical}</Badge>
              <Badge variant="secondary">{healthStats.errors.warning}</Badge>
              <Badge variant="outline">{healthStats.errors.info}</Badge>
            </div>
          </CardContent>
        </Card>
      </div>

      {/* Alerts Table */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <AlertCircle className="w-5 h-5" />
            Active Alerts
          </CardTitle>
        </CardHeader>
        <CardContent className="p-0">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Alert ID</TableHead>
                <TableHead>Asset/Site</TableHead>
                <TableHead>Type</TableHead>
                <TableHead>Severity</TableHead>
                <TableHead>Triggered</TableHead>
                <TableHead>State</TableHead>
                <TableHead>Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {mockAlerts.map((alert) => (
                <TableRow key={alert.id} data-testid={`row-alert-${alert.id}`}>
                  <TableCell className="font-medium">{alert.id}</TableCell>
                  <TableCell>{alert.asset}</TableCell>
                  <TableCell>{alert.type}</TableCell>
                  <TableCell>
                    <Badge variant={alert.severity === "critical" ? "destructive" : "secondary"}>
                      {alert.severity}
                    </Badge>
                  </TableCell>
                  <TableCell>{alert.triggered}</TableCell>
                  <TableCell>
                    <Badge variant="outline">{alert.state}</Badge>
                  </TableCell>
                  <TableCell>
                    <Button variant="ghost" size="sm" data-testid={`button-playbook-${alert.id}`}>
                      Playbook
                    </Button>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </CardContent>
      </Card>
    </div>
  );
}

// Grid Ops Tab Component
function GridOpsTab() {
  const realtimeData = Array.from({ length: 12 }, (_, i) => ({
    time: `${i * 2}:00`,
    actual: Math.random() * 80 + 20,
    forecast: Math.random() * 70 + 30
  }));

  return (
    <div className="space-y-4">
      {/* Real-time Panel */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Radio className="w-5 h-5" />
            Real-time Grid Operations
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="space-y-2">
              <Label>Global Cap (MW)</Label>
              <Slider defaultValue={[50]} max={100} step={1} />
            </div>
            <div className="space-y-2">
              <Label>Dispatch Target (MW)</Label>
              <Input type="number" defaultValue={45} />
            </div>
            <div className="flex items-end">
              <Button className="w-full" data-testid="button-apply-dispatch">
                <Zap className="w-4 h-4 mr-2" />
                Apply Dispatch
              </Button>
            </div>
          </div>

          <div className="h-64">
            <ResponsiveContainer width="100%" height="100%">
              <LineChart data={realtimeData}>
                <CartesianGrid strokeDasharray="3 3" />
                <XAxis dataKey="time" />
                <YAxis />
                <Tooltip />
                <Line type="monotone" dataKey="actual" stroke="#8884d8" strokeWidth={2} />
                <Line type="monotone" dataKey="forecast" stroke="#82ca9d" strokeDasharray="5 5" />
              </LineChart>
            </ResponsiveContainer>
          </div>
        </CardContent>
      </Card>

      {/* Event Manager */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Target className="w-5 h-5" />
            Event Manager
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="space-y-2">
              <Label>Event Start</Label>
              <Input type="datetime-local" />
            </div>
            <div className="space-y-2">
              <Label>Event End</Label>
              <Input type="datetime-local" />
            </div>
            <div className="flex items-end">
              <Button variant="outline" className="w-full" data-testid="button-arm-event">
                Arm Event
              </Button>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}

// Pricing & Billing Tab Component
function PricingBillingTab() {
  const mockTariffs = [
    { id: "TAR-001", name: "Standard Charging", charging: 0.35, v2gCredit: 0.45, scope: "Global", effectiveDate: "2024-01-01" },
    { id: "TAR-002", name: "Premium Sites", charging: 0.42, v2gCredit: 0.55, scope: "Marina Bay, Orchard", effectiveDate: "2024-01-15" },
  ];

  return (
    <div className="space-y-4">
      {/* Tariffs */}
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center justify-between">
            <div className="flex items-center gap-2">
              <DollarSign className="w-5 h-5" />
              Tariffs
            </div>
            <Button size="sm" data-testid="button-create-tariff">
              Create Tariff
            </Button>
          </CardTitle>
        </CardHeader>
        <CardContent className="p-0">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Tariff ID</TableHead>
                <TableHead>Name</TableHead>
                <TableHead>Charging ($/kWh)</TableHead>
                <TableHead>V2G Credit ($/kWh)</TableHead>
                <TableHead>Scope</TableHead>
                <TableHead>Effective Date</TableHead>
                <TableHead>Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {mockTariffs.map((tariff) => (
                <TableRow key={tariff.id} data-testid={`row-tariff-${tariff.id}`}>
                  <TableCell className="font-medium">{tariff.id}</TableCell>
                  <TableCell>{tariff.name}</TableCell>
                  <TableCell>${tariff.charging}</TableCell>
                  <TableCell>${tariff.v2gCredit}</TableCell>
                  <TableCell>{tariff.scope}</TableCell>
                  <TableCell>{tariff.effectiveDate}</TableCell>
                  <TableCell>
                    <Button variant="ghost" size="sm">Edit</Button>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      {/* Billing Summary */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">Total Revenue</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="stat-revenue">$45,230</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">Credits Issued</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold" data-testid="stat-credits">$12,450</div>
          </CardContent>
        </Card>
        <Card>
          <CardHeader className="pb-2">
            <CardTitle className="text-sm font-medium text-muted-foreground">Net Margin</CardTitle>
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-500" data-testid="stat-margin">$32,780</div>
          </CardContent>
        </Card>
      </div>
    </div>
  );
}

// Incentives Programs Tab Component
function IncentivesTab() {
  const mockPrograms = [
    { id: "INC-001", name: "Welcome Bonus", type: "welcome", status: "active", enrollments: 145, conversions: 98, creditsIssued: 14500 },
    { id: "INC-002", name: "Peak Hour Challenge", type: "threshold", status: "active", enrollments: 234, conversions: 187, creditsIssued: 28050 },
    { id: "INC-003", name: "Refer a Friend", type: "referral", status: "scheduled", enrollments: 67, conversions: 23, creditsIssued: 3450 },
  ];

  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold">Incentive Programs</h3>
        <Button data-testid="button-create-program">
          <Gift className="w-4 h-4 mr-2" />
          Create Program
        </Button>
      </div>

      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        {mockPrograms.map((program) => (
          <Card key={program.id} className="hover-elevate" data-testid={`card-program-${program.id}`}>
            <CardHeader>
              <CardTitle className="flex items-center justify-between">
                <span className="text-base">{program.name}</span>
                <Badge variant={program.status === "active" ? "default" : "secondary"}>
                  {program.status}
                </Badge>
              </CardTitle>
            </CardHeader>
            <CardContent className="space-y-2">
              <div className="flex justify-between text-sm">
                <span className="text-muted-foreground">Enrollments:</span>
                <span className="font-medium">{program.enrollments}</span>
              </div>
              <div className="flex justify-between text-sm">
                <span className="text-muted-foreground">Conversions:</span>
                <span className="font-medium">{program.conversions}</span>
              </div>
              <div className="flex justify-between text-sm">
                <span className="text-muted-foreground">Credits Issued:</span>
                <span className="font-medium">${program.creditsIssued}</span>
              </div>
              <Button variant="outline" size="sm" className="w-full mt-2">
                View Details
              </Button>
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  );
}

// Hot Zones Tab Component
function HotZonesTab() {
  return (
    <div className="space-y-4">
      <Card>
        <CardHeader>
          <CardTitle className="flex items-center gap-2">
            <Flame className="w-5 h-5" />
            Demand Hotspots
          </CardTitle>
        </CardHeader>
        <CardContent className="space-y-4">
          <div className="flex gap-4">
            <Select defaultValue="demand">
              <SelectTrigger className="w-48">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="demand">Demand Risk</SelectItem>
                <SelectItem value="price">Price Premium</SelectItem>
                <SelectItem value="queue">Queue Length</SelectItem>
              </SelectContent>
            </Select>
            <Select defaultValue="now">
              <SelectTrigger className="w-48">
                <SelectValue />
              </SelectTrigger>
              <SelectContent>
                <SelectItem value="now">Now</SelectItem>
                <SelectItem value="1h">+1 Hour</SelectItem>
                <SelectItem value="6h">+6 Hours</SelectItem>
              </SelectContent>
            </Select>
          </div>

          <div className="h-96 bg-secondary/30 rounded-lg flex items-center justify-center">
            <p className="text-muted-foreground">Heatmap visualization area</p>
          </div>

          <div className="flex gap-2">
            <Button data-testid="button-push-offer">
              <Send className="w-4 h-4 mr-2" />
              Push Targeted Offer
            </Button>
            <Button variant="outline" data-testid="button-schedule-nudge">
              <Calendar className="w-4 h-4 mr-2" />
              Schedule Nudges
            </Button>
            <Button variant="outline" data-testid="button-export-report">
              <Download className="w-4 h-4 mr-2" />
              Export Report
            </Button>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}

// Notifications Tab Component
function NotificationsTab() {
  const mockTemplates = [
    { id: "TPL-001", name: "Charging Complete", type: "transactional", language: "EN", lastSent: "2hr ago", sent: 234, opened: 198 },
    { id: "TPL-002", name: "V2G Event Alert", type: "operational", language: "EN", lastSent: "1day ago", sent: 567, opened: 489 },
    { id: "TPL-003", name: "Monthly Summary", type: "marketing", language: "EN", lastSent: "5days ago", sent: 1203, opened: 856 },
  ];

  return (
    <div className="space-y-4">
      <div className="flex justify-between items-center">
        <h3 className="text-lg font-semibold">Notification Templates</h3>
        <Button data-testid="button-create-template">
          <Bell className="w-4 h-4 mr-2" />
          Create Template
        </Button>
      </div>

      <Card>
        <CardContent className="p-0">
          <Table>
            <TableHeader>
              <TableRow>
                <TableHead>Template ID</TableHead>
                <TableHead>Name</TableHead>
                <TableHead>Type</TableHead>
                <TableHead>Language</TableHead>
                <TableHead>Last Sent</TableHead>
                <TableHead>Sent</TableHead>
                <TableHead>Opened</TableHead>
                <TableHead>Actions</TableHead>
              </TableRow>
            </TableHeader>
            <TableBody>
              {mockTemplates.map((template) => (
                <TableRow key={template.id} data-testid={`row-template-${template.id}`}>
                  <TableCell className="font-medium">{template.id}</TableCell>
                  <TableCell>{template.name}</TableCell>
                  <TableCell>
                    <Badge variant="outline">{template.type}</Badge>
                  </TableCell>
                  <TableCell>{template.language}</TableCell>
                  <TableCell>{template.lastSent}</TableCell>
                  <TableCell>{template.sent}</TableCell>
                  <TableCell>{template.opened}</TableCell>
                  <TableCell>
                    <div className="flex gap-2">
                      <Button variant="ghost" size="sm">Edit</Button>
                      <Button variant="ghost" size="sm">Test</Button>
                    </div>
                  </TableCell>
                </TableRow>
              ))}
            </TableBody>
          </Table>
        </CardContent>
      </Card>

      {/* Compliance Section */}
      <Card>
        <CardHeader>
          <CardTitle>Compliance & Audit</CardTitle>
        </CardHeader>
        <CardContent>
          <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
            <div className="space-y-1">
              <p className="text-sm text-muted-foreground">Opt-in Rate</p>
              <p className="text-2xl font-bold">87.3%</p>
            </div>
            <div className="space-y-1">
              <p className="text-sm text-muted-foreground">Opt-out Rate</p>
              <p className="text-2xl font-bold">12.7%</p>
            </div>
            <div className="space-y-1">
              <p className="text-sm text-muted-foreground">Total Sends (30d)</p>
              <p className="text-2xl font-bold">45,678</p>
            </div>
          </div>
        </CardContent>
      </Card>
    </div>
  );
}

// Main Asset Management Component with Tabs
export default function AssetManagement() {
  return (
    <div className="p-6">
      <div className="mb-6">
        <h1 className="text-3xl font-bold">Asset Management</h1>
        <p className="text-muted-foreground">Manage all V2G infrastructure assets, sites, and operations</p>
      </div>

      <Tabs defaultValue="chargers" className="space-y-4">
        <TabsList className="grid grid-cols-3 lg:grid-cols-9 w-full">
          <TabsTrigger value="chargers" data-testid="tab-chargers">
            <Battery className="w-4 h-4 mr-2" />
            Chargers
          </TabsTrigger>
          <TabsTrigger value="sites" data-testid="tab-sites">
            <Building2 className="w-4 h-4 mr-2" />
            Sites
          </TabsTrigger>
          <TabsTrigger value="vehicles" data-testid="tab-vehicles">
            <Car className="w-4 h-4 mr-2" />
            Vehicles
          </TabsTrigger>
          <TabsTrigger value="health" data-testid="tab-health">
            <Activity className="w-4 h-4 mr-2" />
            Health
          </TabsTrigger>
          <TabsTrigger value="gridops" data-testid="tab-gridops">
            <Radio className="w-4 h-4 mr-2" />
            Grid Ops
          </TabsTrigger>
          <TabsTrigger value="pricing" data-testid="tab-pricing">
            <DollarSign className="w-4 h-4 mr-2" />
            Pricing
          </TabsTrigger>
          <TabsTrigger value="incentives" data-testid="tab-incentives">
            <Gift className="w-4 h-4 mr-2" />
            Programs
          </TabsTrigger>
          <TabsTrigger value="hotzones" data-testid="tab-hotzones">
            <Flame className="w-4 h-4 mr-2" />
            Hot Zones
          </TabsTrigger>
          <TabsTrigger value="notifications" data-testid="tab-notifications">
            <Bell className="w-4 h-4 mr-2" />
            Alerts
          </TabsTrigger>
        </TabsList>

        <TabsContent value="chargers">
          <ChargersTab />
        </TabsContent>

        <TabsContent value="sites">
          <SitesTab />
        </TabsContent>

        <TabsContent value="vehicles">
          <VehiclesTab />
        </TabsContent>

        <TabsContent value="health">
          <HealthAlertsTab />
        </TabsContent>

        <TabsContent value="gridops">
          <GridOpsTab />
        </TabsContent>

        <TabsContent value="pricing">
          <PricingBillingTab />
        </TabsContent>

        <TabsContent value="incentives">
          <IncentivesTab />
        </TabsContent>

        <TabsContent value="hotzones">
          <HotZonesTab />
        </TabsContent>

        <TabsContent value="notifications">
          <NotificationsTab />
        </TabsContent>
      </Tabs>
    </div>
  );
}
