import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Progress } from "@/components/ui/progress";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { useState } from "react";
import { 
  Car, 
  Battery, 
  Wifi, 
  WifiOff, 
  MapPin, 
  Clock, 
  User,
  Zap,
  Power,
  Users,
  Activity,
  Bell,
  UserPlus,
  TrendingUp,
  Phone
} from "lucide-react";

interface VehicleProfile {
  id: string;
  name: string;
  type: "sedan" | "suv" | "truck" | "bus" | "van";
  batteryCapacity: number; // kWh
  currentSOC: number; // %
  maxDischargeRate: number; // kW
  status: "charging" | "discharging" | "idle" | "offline";
  isConnected: boolean;
  location: {
    lat: number;
    lng: number;
    address: string;
    district: string;
  };
  connectedDuration: number; // minutes
  sessionStartTime: string;
  powerAvailable: number; // kW available to grid
  chargeCycles: number; // Number of charge cycles
  owner: {
    name: string;
    id: string;
    avatar?: string;
    plan: "basic" | "premium" | "fleet";
    joinDate: string;
  };
  lastActivity: string;
  totalDischarges: number;
  totalEarnings: number;
}

// Mock vehicle profiles - Updated with user requirements
const mockVehicleProfiles: VehicleProfile[] = [
  {
    id: "EV-101",
    name: "Tesla Model S",
    type: "sedan",
    batteryCapacity: 100,
    currentSOC: 82,
    maxDischargeRate: 75,
    status: "discharging",
    isConnected: true,
    location: {
      lat: 1.3521,
      lng: 103.8198,
      address: "Marina Bay Charging Hub",
      district: "Central Business District"
    },
    connectedDuration: 45,
    sessionStartTime: "14:30",
    powerAvailable: 48,
    chargeCycles: 12,
    owner: {
      name: "Sarah Chen",
      id: "user-001",
      avatar: "/avatar1.jpg",
      plan: "premium",
      joinDate: "2024-01-15"
    },
    lastActivity: "2 min ago",
    totalDischarges: 3,
    totalEarnings: 850.50
  },
  {
    id: "EV-202",
    name: "BMW i4",
    type: "sedan", 
    batteryCapacity: 84,
    currentSOC: 78,
    maxDischargeRate: 75,
    status: "charging",
    isConnected: true,
    location: {
      lat: 1.3048,
      lng: 103.8318,
      address: "Orchard Plaza Station",
      district: "Orchard Shopping District"
    },
    connectedDuration: 52,
    sessionStartTime: "12:00",
    powerAvailable: 42,
    chargeCycles: 6,
    owner: {
      name: "Marcus Wong",
      id: "user-002",
      plan: "basic",
      joinDate: "2024-03-20"
    },
    lastActivity: "5 min ago",
    totalDischarges: 2,
    totalEarnings: 640.25
  },
  {
    id: "EV-303",
    name: "Nissan Leaf",
    type: "sedan",
    batteryCapacity: 62,
    currentSOC: 81,
    maxDischargeRate: 75,
    status: "idle",
    isConnected: true,
    location: {
      lat: 1.3294,
      lng: 103.7414,
      address: "Jurong Point Mall",
      district: "Jurong Industrial Estate"
    },
    connectedDuration: 30,
    sessionStartTime: "15:00",
    powerAvailable: 45,
    chargeCycles: 3,
    owner: {
      name: "Lisa Tan",
      id: "user-003",
      plan: "basic",
      joinDate: "2024-02-10"
    },
    lastActivity: "1 min ago",
    totalDischarges: 4,
    totalEarnings: 925.75
  },
  {
    id: "EV-404",
    name: "Volvo XC40",
    type: "suv",
    batteryCapacity: 78,
    currentSOC: 79,
    maxDischargeRate: 75,
    status: "discharging",
    isConnected: true,
    location: {
      lat: 1.3383,
      lng: 103.8441,
      address: "Toa Payoh Hub",
      district: "Toa Payoh"
    },
    connectedDuration: 58,
    sessionStartTime: "16:00",
    powerAvailable: 50,
    chargeCycles: 8,
    owner: {
      name: "Fleet Services Ltd",
      id: "fleet-001",
      plan: "fleet",
      joinDate: "2023-11-05"
    },
    lastActivity: "25 min ago",
    totalDischarges: 3,
    totalEarnings: 780.00
  },
  {
    id: "EV-505",
    name: "Audi e-tron",
    type: "suv",
    batteryCapacity: 95,
    currentSOC: 68,
    maxDischargeRate: 75,
    status: "idle",
    isConnected: true,
    location: {
      lat: 1.3200,
      lng: 103.8500,
      address: "Bugis Junction",
      district: "Bugis"
    },
    connectedDuration: 35,
    sessionStartTime: "13:45",
    powerAvailable: 52,
    chargeCycles: 15,
    owner: {
      name: "David Lim",
      id: "user-005",
      plan: "premium",
      joinDate: "2024-01-20"
    },
    lastActivity: "8 min ago",
    totalDischarges: 5,
    totalEarnings: 1120.75
  },
  {
    id: "EV-606",
    name: "Mercedes EQC",
    type: "suv",
    batteryCapacity: 80,
    currentSOC: 92,
    maxDischargeRate: 75,
    status: "charging",
    isConnected: true,
    location: {
      lat: 1.2800,
      lng: 103.8400,
      address: "Sentosa Gateway",
      district: "Sentosa"
    },
    connectedDuration: 25,
    sessionStartTime: "14:50",
    powerAvailable: 38,
    chargeCycles: 4,
    owner: {
      name: "Rachel Koh",
      id: "user-006",
      plan: "basic",
      joinDate: "2024-04-10"
    },
    lastActivity: "15 min ago",
    totalDischarges: 1,
    totalEarnings: 245.50
  },
  {
    id: "EV-707",
    name: "BYD Atto 3",
    type: "suv",
    batteryCapacity: 60,
    currentSOC: 55,
    maxDischargeRate: 75,
    status: "idle",
    isConnected: true,
    location: {
      lat: 1.3700,
      lng: 103.8900,
      address: "Paya Lebar Quarter",
      district: "Paya Lebar"
    },
    connectedDuration: 42,
    sessionStartTime: "11:30",
    powerAvailable: 28,
    chargeCycles: 9,
    owner: {
      name: "James Ng",
      id: "user-007",
      plan: "basic",
      joinDate: "2024-02-28"
    },
    lastActivity: "3 min ago",
    totalDischarges: 3,
    totalEarnings: 680.25
  },
  {
    id: "EV-808",
    name: "Hyundai Ioniq 6",
    type: "sedan",
    batteryCapacity: 77,
    currentSOC: 85,
    maxDischargeRate: 75,
    status: "discharging",
    isConnected: true,
    location: {
      lat: 1.3100,
      lng: 103.7800,
      address: "Clementi Mall",
      district: "Clementi"
    },
    connectedDuration: 48,
    sessionStartTime: "10:15",
    powerAvailable: 55,
    chargeCycles: 11,
    owner: {
      name: "Emily Tan",
      id: "user-008",
      plan: "premium",
      joinDate: "2024-01-05"
    },
    lastActivity: "6 min ago",
    totalDischarges: 4,
    totalEarnings: 955.00
  },
  {
    id: "EV-909",
    name: "Polestar 2",
    type: "sedan",
    batteryCapacity: 78,
    currentSOC: 72,
    maxDischargeRate: 75,
    status: "charging",
    isConnected: true,
    location: {
      lat: 1.3500,
      lng: 103.9400,
      address: "Tampines Hub",
      district: "Tampines"
    },
    connectedDuration: 38,
    sessionStartTime: "13:20",
    powerAvailable: 44,
    chargeCycles: 7,
    owner: {
      name: "Kevin Lee",
      id: "user-009",
      plan: "basic",
      joinDate: "2024-03-15"
    },
    lastActivity: "10 min ago",
    totalDischarges: 2,
    totalEarnings: 520.00
  },
  {
    id: "EV-1010",
    name: "Kia EV6",
    type: "suv",
    batteryCapacity: 77,
    currentSOC: 64,
    maxDischargeRate: 75,
    status: "idle",
    isConnected: true,
    location: {
      lat: 1.4400,
      lng: 103.7900,
      address: "Woodlands Checkpoint",
      district: "Woodlands"
    },
    connectedDuration: 55,
    sessionStartTime: "09:45",
    powerAvailable: 46,
    chargeCycles: 13,
    owner: {
      name: "Green Transport Co",
      id: "fleet-002",
      plan: "fleet",
      joinDate: "2023-12-01"
    },
    lastActivity: "20 min ago",
    totalDischarges: 6,
    totalEarnings: 1340.50
  }
];

const getStatusColor = (status: string) => {
  switch (status) {
    case "charging": return "bg-green-500";
    case "discharging": return "bg-blue-500";
    case "idle": return "bg-yellow-500";
    case "offline": return "bg-gray-500";
    default: return "bg-gray-500";
  }
};

const getStatusIcon = (status: string) => {
  switch (status) {
    case "charging": return <Battery className="h-4 w-4" />;
    case "discharging": return <Zap className="h-4 w-4" />;
    case "idle": return <Clock className="h-4 w-4" />;
    case "offline": return <WifiOff className="h-4 w-4" />;
    default: return <Activity className="h-4 w-4" />;
  }
};

const getVehicleTypeIcon = (type: string) => {
  return <Car className="h-4 w-4" />; // Could be expanded with different icons per type
};

const getPlanColor = (plan: string) => {
  switch (plan) {
    case "premium": return "bg-purple-500";
    case "fleet": return "bg-blue-500";
    case "basic": return "bg-gray-500";
    default: return "bg-gray-500";
  }
};

export default function VehiclesDashboard() {
  const [chargeCycleFilter, setChargeCycleFilter] = useState<string>("all");
  const [notificationFilter, setNotificationFilter] = useState<string>("all");
  
  const vehicleCalls = 40;
  const powerMadeAvailable = 2; // MW
  const averageSOC = 80;
  const activeSessions = 40;
  const driversSignedUp = 1000;
  const totalCapacityAdded = 3000; // kW
  const newDischargesMadeAvailable = 120;
  const gridCallsMade = 85;
  const gridCallsAccepted = 68;

  const getFilteredVehicles = () => {
    if (chargeCycleFilter === "all") return mockVehicleProfiles;
    const cycleCount = parseInt(chargeCycleFilter);
    return mockVehicleProfiles.filter((v) => v.chargeCycles >= cycleCount);
  };

  const filteredVehicles = getFilteredVehicles();

  const getNotificationsByFilter = () => {
    if (notificationFilter === "all") {
      return { label: "All Notifications", made: gridCallsMade, accepted: gridCallsAccepted };
    } else if (notificationFilter === "user") {
      return { label: "By User", made: 32, accepted: 25 };
    } else if (notificationFilter === "location") {
      return { label: "By Location/District", made: 53, accepted: 43 };
    }
    return { label: "All Notifications", made: gridCallsMade, accepted: gridCallsAccepted };
  };

  const notificationData = getNotificationsByFilter();

  return (
    <div className="p-6 space-y-6" data-testid="page-vehicles-dashboard">
      {/* Header with Notification Button */}
      <div className="flex justify-between items-start">
        <div className="space-y-2">
          <h1 className="text-3xl font-bold text-foreground">Grid Dashboard</h1>
          <p className="text-muted-foreground">
            Monitor connected vehicles, power availability, and user profiles
          </p>
        </div>
        <div className="flex gap-2 items-start">
          <Select value={notificationFilter} onValueChange={setNotificationFilter}>
            <SelectTrigger className="w-[200px]" data-testid="select-notification-filter">
              <SelectValue placeholder="Filter notifications" />
            </SelectTrigger>
            <SelectContent>
              <SelectItem value="all">All Notifications</SelectItem>
              <SelectItem value="user">By User</SelectItem>
              <SelectItem value="location">By Location/District</SelectItem>
            </SelectContent>
          </Select>
          
          <Button 
            variant="outline" 
            className="relative flex flex-col items-start py-3 h-auto"
            data-testid="button-notifications"
          >
            <div className="flex items-center gap-2 w-full">
              <Bell className="h-4 w-4" />
              <span className="font-semibold">{notificationData.label}</span>
              <Badge className="bg-red-500 text-white" data-testid="badge-notification-count">
                {notificationData.made - notificationData.accepted}
              </Badge>
            </div>
            <div className="text-xs text-muted-foreground mt-1 flex gap-3">
              <span>Calls Made: {notificationData.made}</span>
              <span>Accepted: {notificationData.accepted}</span>
            </div>
          </Button>
        </div>
      </div>

      {/* Top Level Metrics */}
      <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
        <Card data-testid="card-drivers-signed">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">No of Drivers Signed</CardTitle>
            <UserPlus className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-blue-600">{driversSignedUp}</div>
            <p className="text-xs text-muted-foreground">
              Registered users
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-total-capacity">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Total Capacity Added</CardTitle>
            <TrendingUp className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{totalCapacityAdded} kW</div>
            <p className="text-xs text-muted-foreground">
              Grid capacity increase
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-new-discharges">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">No of New Discharges Made Available</CardTitle>
            <Zap className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-purple-600">{newDischargesMadeAvailable}</div>
            <p className="text-xs text-muted-foreground">
              Available sessions
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Overview Stats */}
      <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
        <Card data-testid="card-vehicle-calls">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">No of Vehicle Calls</CardTitle>
            <Phone className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold text-green-600">{vehicleCalls}</div>
            <p className="text-xs text-muted-foreground">
              Grid requests made
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-power-made-available">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Power Made Available</CardTitle>
            <Power className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{powerMadeAvailable} MW</div>
            <p className="text-xs text-muted-foreground">
              Ready for grid discharge
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-average-soc">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Average SOC</CardTitle>
            <Battery className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{averageSOC}%</div>
            <p className="text-xs text-muted-foreground">
              Fleet battery level
            </p>
          </CardContent>
        </Card>

        <Card data-testid="card-active-sessions">
          <CardHeader className="flex flex-row items-center justify-between space-y-0 pb-2">
            <CardTitle className="text-sm font-medium">Active Sessions</CardTitle>
            <Users className="h-4 w-4 text-muted-foreground" />
          </CardHeader>
          <CardContent>
            <div className="text-2xl font-bold">{activeSessions}</div>
            <p className="text-xs text-muted-foreground">
              Currently engaged
            </p>
          </CardContent>
        </Card>
      </div>

      {/* Vehicle Profiles List */}
      <Card data-testid="card-vehicle-profiles">
        <CardHeader>
          <div className="flex justify-between items-center">
            <CardTitle className="flex items-center gap-2">
              <Car className="h-5 w-5" />
              Vehicle Fleet Overview
            </CardTitle>
            <div className="flex items-center gap-2">
              <span className="text-sm text-muted-foreground">Filter by charge cycles:</span>
              <Select value={chargeCycleFilter} onValueChange={setChargeCycleFilter}>
                <SelectTrigger className="w-[200px]" data-testid="select-charge-cycle-filter">
                  <SelectValue placeholder="All vehicles" />
                </SelectTrigger>
                <SelectContent>
                  <SelectItem value="all">All vehicles</SelectItem>
                  <SelectItem value="3">3 or more cycles</SelectItem>
                  <SelectItem value="5">5 or more cycles</SelectItem>
                  <SelectItem value="8">8 or more cycles</SelectItem>
                  <SelectItem value="10">10 or more cycles</SelectItem>
                </SelectContent>
              </Select>
            </div>
          </div>
        </CardHeader>
        <CardContent>
          <div className="space-y-4">
            {filteredVehicles.map((vehicle) => (
              <div 
                key={vehicle.id} 
                className="border rounded-lg p-4 hover-elevate"
                data-testid={`vehicle-${vehicle.id}`}
              >
                <div className="flex justify-between items-start mb-4">
                  <div className="flex items-center gap-3">
                    <div className="flex items-center gap-2">
                      {getVehicleTypeIcon(vehicle.type)}
                      <div>
                        <h3 className="font-semibold text-lg">{vehicle.name}</h3>
                        <div className="text-sm text-muted-foreground">{vehicle.id}</div>
                      </div>
                    </div>
                  </div>
                  
                  <div className="flex items-center gap-2">
                    <Badge 
                      className={`${getStatusColor(vehicle.status)} text-white flex items-center gap-1`}
                      data-testid={`status-${vehicle.status}`}
                    >
                      {getStatusIcon(vehicle.status)}
                      {vehicle.status.toUpperCase()}
                    </Badge>
                    {vehicle.isConnected ? (
                      <Wifi className="h-4 w-4 text-green-500" />
                    ) : (
                      <WifiOff className="h-4 w-4 text-red-500" />
                    )}
                  </div>
                </div>

                <div className="grid grid-cols-1 md:grid-cols-4 gap-4 mb-4">
                  <div className="space-y-2">
                    <div className="text-sm font-medium">Battery Status</div>
                    <div className="text-xl font-bold">{vehicle.currentSOC}%</div>
                    <Progress value={vehicle.currentSOC} className="h-2" />
                    <div className="text-xs text-muted-foreground">
                      {vehicle.batteryCapacity} kWh capacity
                    </div>
                  </div>

                  <div className="space-y-2">
                    <div className="text-sm font-medium">Power Available</div>
                    <div className="text-xl font-bold text-blue-600">{vehicle.powerAvailable} kW</div>
                    <div className="text-xs text-muted-foreground">
                      Max: {vehicle.maxDischargeRate} kW
                    </div>
                  </div>

                  <div className="space-y-2">
                    <div className="text-sm font-medium">Session Duration</div>
                    <div className="text-xl font-bold">{vehicle.connectedDuration} min</div>
                    <div className="text-xs text-muted-foreground">
                      Started: {vehicle.sessionStartTime}
                    </div>
                  </div>

                  <div className="space-y-2">
                    <div className="text-sm font-medium">Location</div>
                    <div className="flex items-center gap-1 text-sm">
                      <MapPin className="h-3 w-3" />
                      <span className="truncate">{vehicle.location.address}</span>
                    </div>
                    <div className="text-xs text-muted-foreground">
                      {vehicle.location.lat.toFixed(4)}, {vehicle.location.lng.toFixed(4)}
                    </div>
                  </div>
                </div>

                {/* User Profile Section */}
                <div className="border-t pt-4">
                  <div className="flex justify-between items-center">
                    <div className="flex items-center gap-3">
                      <Avatar className="h-10 w-10">
                        <AvatarImage src={vehicle.owner.avatar} />
                        <AvatarFallback>
                          {vehicle.owner.name.split(' ').map(n => n[0]).join('')}
                        </AvatarFallback>
                      </Avatar>
                      <div>
                        <div className="font-medium">{vehicle.owner.name}</div>
                        <div className="text-sm text-muted-foreground flex items-center gap-2">
                          <Badge 
                            className={`${getPlanColor(vehicle.owner.plan)} text-white text-xs`}
                            data-testid={`plan-${vehicle.owner.plan}`}
                          >
                            {vehicle.owner.plan.toUpperCase()}
                          </Badge>
                          <span>Last active: {vehicle.lastActivity}</span>
                        </div>
                      </div>
                    </div>

                    <div className="text-right space-y-1">
                      <div className="text-sm text-muted-foreground">Total Earnings</div>
                      <div className="text-lg font-bold text-green-600">
                        ${vehicle.totalEarnings.toFixed(2)}
                      </div>
                      <div className="text-xs text-muted-foreground">
                        {vehicle.totalDischarges} of max 12 discharges
                      </div>
                    </div>

                    <Button 
                      variant={vehicle.isConnected ? "default" : "outline"}
                      disabled={!vehicle.isConnected}
                      data-testid={`action-${vehicle.id}`}
                    >
                      {vehicle.status === "discharging" ? "Stop Discharge" : "Start Discharge"}
                    </Button>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </CardContent>
      </Card>
    </div>
  );
}