// from blueprint:javascript_database
import { 
  users,
  v2gSessions,
  type User, 
  type InsertUser,
  type SelectNotificationSubscription,
  type InsertNotificationSubscription,
  type SelectPing,
  type InsertPing,
  type InsertV2GSession,
  type V2GSession
} from "@shared/schema";
import { db } from "./db";
import { eq, and, count } from "drizzle-orm";
import { randomUUID } from "crypto";
import session from "express-session";
import connectPgSimple from "connect-pg-simple";
import { pool } from "./db";

const PgSession = connectPgSimple(session);

// modify the interface with any CRUD methods
// you might need

export interface IStorage {
  getUser(id: string): Promise<User | undefined>;
  getUserByUsername(username: string): Promise<User | undefined>;
  createUser(user: InsertUser): Promise<User>;
  
  // V2G Sessions - Max 5 per user, Max 12 per manufacturer
  createV2GSession(session: InsertV2GSession): Promise<V2GSession>;
  getActiveV2GSessionsByUser(userId: string): Promise<V2GSession[]>;
  getActiveV2GSessionsByManufacturer(manufacturer: string): Promise<V2GSession[]>;
  getUserV2GSessionCount(userId: string): Promise<number>;
  getManufacturerV2GSessionCount(manufacturer: string): Promise<number>;
  completeV2GSession(sessionId: string, endTime: string, incentiveEarned: number): Promise<V2GSession>;
  
  // Notification subscriptions
  createNotificationSubscription(subscription: InsertNotificationSubscription): Promise<SelectNotificationSubscription>;
  getNotificationSubscription(userId: string): Promise<SelectNotificationSubscription | undefined>;
  deleteNotificationSubscription(userId: string): Promise<boolean>;
  getAllNotificationSubscriptions(): Promise<SelectNotificationSubscription[]>;
  
  // Ping notifications
  createPing(ping: InsertPing): Promise<SelectPing>;
  getPingsByUserId(userId: string): Promise<SelectPing[]>;
  getPingsByRegion(region: string): Promise<SelectPing[]>;
  getRecentPings(since: Date): Promise<SelectPing[]>;
  
  // Session store for authentication - from blueprint:javascript_auth_all_persistance
  sessionStore: session.Store;
}

// from blueprint:javascript_database - use database for users and sessions
export class DatabaseStorage implements IStorage {
  private notificationSubscriptions: Map<string, SelectNotificationSubscription>;
  private pings: Map<string, SelectPing>;
  public sessionStore: session.Store;

  constructor() {
    this.notificationSubscriptions = new Map();
    this.pings = new Map();
    // Use PostgreSQL for persistent session storage
    this.sessionStore = new PgSession({
      pool: pool,
      tableName: 'session',
      createTableIfMissing: true,
    });
  }

  async getUser(id: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.id, id));
    return user || undefined;
  }

  async getUserByUsername(username: string): Promise<User | undefined> {
    const [user] = await db.select().from(users).where(eq(users.username, username));
    return user || undefined;
  }

  async createUser(insertUser: InsertUser): Promise<User> {
    const [user] = await db
      .insert(users)
      .values(insertUser)
      .returning();
    return user;
  }

  // V2G Session methods - enforce limits
  async createV2GSession(insertSession: InsertV2GSession): Promise<V2GSession> {
    const [session] = await db
      .insert(v2gSessions)
      .values(insertSession)
      .returning();
    return session;
  }

  async getActiveV2GSessionsByUser(userId: string): Promise<V2GSession[]> {
    return await db
      .select()
      .from(v2gSessions)
      .where(and(eq(v2gSessions.userId, userId), eq(v2gSessions.status, 'active')));
  }

  async getActiveV2GSessionsByManufacturer(manufacturer: string): Promise<V2GSession[]> {
    return await db
      .select()
      .from(v2gSessions)
      .where(and(eq(v2gSessions.manufacturer, manufacturer), eq(v2gSessions.status, 'active')));
  }

  async getUserV2GSessionCount(userId: string): Promise<number> {
    const result = await db
      .select({ count: count() })
      .from(v2gSessions)
      .where(and(eq(v2gSessions.userId, userId), eq(v2gSessions.status, 'active')));
    return result[0]?.count || 0;
  }

  async getManufacturerV2GSessionCount(manufacturer: string): Promise<number> {
    const result = await db
      .select({ count: count() })
      .from(v2gSessions)
      .where(and(eq(v2gSessions.manufacturer, manufacturer), eq(v2gSessions.status, 'active')));
    return result[0]?.count || 0;
  }

  async completeV2GSession(sessionId: string, endTime: string, incentiveEarned: number): Promise<V2GSession> {
    const [session] = await db
      .update(v2gSessions)
      .set({ endTime, incentiveEarned, status: 'completed' })
      .where(eq(v2gSessions.id, sessionId))
      .returning();
    return session;
  }

  // Notification subscription methods
  async createNotificationSubscription(insertSubscription: InsertNotificationSubscription): Promise<SelectNotificationSubscription> {
    const id = randomUUID();
    const subscription: SelectNotificationSubscription = {
      ...insertSubscription,
      id,
      createdAt: new Date().toISOString()
    };
    this.notificationSubscriptions.set(insertSubscription.userId, subscription);
    return subscription;
  }

  async getNotificationSubscription(userId: string): Promise<SelectNotificationSubscription | undefined> {
    return this.notificationSubscriptions.get(userId);
  }

  async deleteNotificationSubscription(userId: string): Promise<boolean> {
    return this.notificationSubscriptions.delete(userId);
  }

  async getAllNotificationSubscriptions(): Promise<SelectNotificationSubscription[]> {
    return Array.from(this.notificationSubscriptions.values());
  }

  // Ping notification methods  
  async createPing(insertPing: InsertPing): Promise<SelectPing> {
    const id = randomUUID();
    const ping: SelectPing = {
      ...insertPing,
      id,
      sentAt: new Date().toISOString(),
      vehicleId: insertPing.vehicleId || null,
      userId: insertPing.userId || null,
      region: insertPing.region || null,
      payload: insertPing.payload || null,
      priority: insertPing.priority || 'normal',
      ttl: insertPing.ttl || 3600
    };
    this.pings.set(id, ping);
    return ping;
  }

  async getPingsByUserId(userId: string): Promise<SelectPing[]> {
    return Array.from(this.pings.values()).filter(ping => ping.userId === userId);
  }

  async getPingsByRegion(region: string): Promise<SelectPing[]> {
    return Array.from(this.pings.values()).filter(ping => ping.region === region);
  }

  async getRecentPings(since: Date): Promise<SelectPing[]> {
    return Array.from(this.pings.values()).filter(ping => 
      new Date(ping.sentAt) >= since
    );
  }
}

export const storage = new DatabaseStorage();
