import { useEffect, useRef, useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { MapPin, DollarSign, Zap, Navigation, Flame, ArrowUpCircle } from "lucide-react";
import type { ChargerLocation } from "@shared/singapore-data";

interface ChargerMapProps {
  chargers: ChargerLocation[];
  selectedCharger?: ChargerLocation;
  onChargerSelect?: (charger: ChargerLocation) => void;
}

const hotspots = [
  { lat: 1.3012, lng: 103.8395, intensity: 0.8, label: "Orchard" },
  { lat: 1.2834, lng: 103.8607, intensity: 0.9, label: "Marina Bay" },
  { lat: 1.3343, lng: 103.9632, intensity: 0.6, label: "Changi" },
];

export function ChargerMap({ chargers, selectedCharger, onChargerSelect }: ChargerMapProps) {
  const [isPeakHour, setIsPeakHour] = useState(false);
  const [showHotspots, setShowHotspots] = useState(false);
  const mapRef = useRef<HTMLDivElement>(null);

  useEffect(() => {
    const hour = new Date().getHours();
    setIsPeakHour(hour >= 18 && hour < 22);
  }, []);

  const getStatusColor = (status: string) => {
    switch (status) {
      case "charging":
        return "bg-primary";
      case "idle":
        return "bg-idle";
      case "offline":
        return "bg-muted";
      default:
        return "bg-muted";
    }
  };

  const currentPrice = (charger: ChargerLocation) =>
    isPeakHour ? charger.peakPricePerKwh : charger.pricePerKwh;

  const singaporeBounds = {
    minLat: 1.15,
    maxLat: 1.47,
    minLng: 103.6,
    maxLng: 104.05,
  };

  const latToY = (lat: number) => {
    const ratio = (lat - singaporeBounds.minLat) / (singaporeBounds.maxLat - singaporeBounds.minLat);
    return (1 - ratio) * 100;
  };

  const lngToX = (lng: number) => {
    const ratio = (lng - singaporeBounds.minLng) / (singaporeBounds.maxLng - singaporeBounds.minLng);
    return ratio * 100;
  };

  return (
    <Card data-testid="card-charger-map">
      <CardHeader>
        <div className="flex items-center justify-between gap-4 flex-wrap">
          <CardTitle className="flex items-center gap-2">
            <MapPin className="h-5 w-5 text-primary" />
            Singapore Charger Network
          </CardTitle>
          <div className="flex items-center gap-2">
            <Button
              variant={showHotspots ? "default" : "outline"}
              size="sm"
              onClick={() => setShowHotspots(!showHotspots)}
              data-testid="button-toggle-hotspots"
            >
              <Flame className="h-4 w-4 mr-2" />
              {showHotspots ? "Hide" : "Show"} Hotspots
            </Button>
            <Badge variant={isPeakHour ? "default" : "outline"} data-testid="badge-map-peak-status">
              {isPeakHour ? "Peak Hours" : "Off-Peak"}
            </Badge>
          </div>
        </div>
      </CardHeader>
      <CardContent>
        <div className="relative w-full aspect-[4/3] bg-muted/20 rounded-lg border border-border overflow-hidden" ref={mapRef}>
          <div className="absolute inset-0 p-4">
            <div className="text-xs text-muted-foreground absolute top-2 left-2 bg-background/80 backdrop-blur-sm px-2 py-1 rounded">
              Singapore Island
            </div>
            
            {showHotspots && hotspots.map((hotspot, idx) => {
              const x = lngToX(hotspot.lng);
              const y = latToY(hotspot.lat);
              const size = 120 * hotspot.intensity;
              const opacity = 0.3 * hotspot.intensity;
              
              return (
                <div
                  key={`hotspot-${idx}`}
                  className="absolute pointer-events-none"
                  style={{
                    left: `${x}%`,
                    top: `${y}%`,
                    transform: 'translate(-50%, -50%)',
                  }}
                  data-testid={`hotspot-${idx}`}
                >
                  <div
                    className="rounded-full bg-primary"
                    style={{
                      width: `${size}px`,
                      height: `${size}px`,
                      opacity: opacity,
                      filter: 'blur(20px)',
                    }}
                  />
                  <div className="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 bg-background/90 backdrop-blur-sm px-2 py-1 rounded text-xs font-semibold whitespace-nowrap">
                    <Flame className="h-3 w-3 inline-block mr-1 text-primary" />
                    {hotspot.label}
                  </div>
                </div>
              );
            })}
            
            {chargers.map((charger) => {
              const x = lngToX(charger.lng);
              const y = latToY(charger.lat);
              const isSelected = selectedCharger?.chargerId === charger.chargerId;
              const price = currentPrice(charger);

              return (
                <div
                  key={charger.chargerId}
                  className="absolute cursor-pointer transform -translate-x-1/2 -translate-y-1/2"
                  style={{ left: `${x}%`, top: `${y}%` }}
                  onClick={() => onChargerSelect?.(charger)}
                  data-testid={`map-pin-${charger.chargerId}`}
                >
                  <div className={`relative group`}>
                    <div
                      className={`w-6 h-6 rounded-full ${getStatusColor(charger.status)} border-2 border-background shadow-lg transition-transform ${
                        isSelected ? "scale-150 ring-2 ring-primary ring-offset-2" : "hover:scale-125"
                      }`}
                    >
                      {charger.status === "charging" && (
                        <Zap className="h-3 w-3 text-background absolute inset-0 m-auto" />
                      )}
                    </div>
                    
                    <div className={`absolute bottom-full left-1/2 transform -translate-x-1/2 mb-2 ${isSelected || 'opacity-0 group-hover:opacity-100'} transition-opacity pointer-events-none`}>
                      <div className="bg-popover border border-border rounded-lg p-2 shadow-lg min-w-[200px]">
                        <div className="font-semibold text-sm truncate">{charger.chargerName}</div>
                        <div className="text-xs text-muted-foreground mb-1">{charger.area}</div>
                        <div className="flex items-center justify-between text-xs mb-1">
                          <span className="capitalize">{charger.status}</span>
                          <span className="font-semibold text-primary">${price.toFixed(2)}/kWh</span>
                        </div>
                        <div className="flex items-center gap-2 flex-wrap">
                          <Badge variant="outline" className="text-xs">
                            {charger.maxPower} kW
                          </Badge>
                          <Badge variant="outline" className="text-xs">
                            {charger.connectorType}
                          </Badge>
                          {charger.v2gCapable && (
                            <Badge variant="default" className="text-xs" data-testid={`v2g-badge-${charger.chargerId}`}>
                              <ArrowUpCircle className="h-3 w-3 mr-1" />
                              V2G
                            </Badge>
                          )}
                        </div>
                        {charger.power && (
                          <div className="text-xs text-primary font-semibold mt-1">
                            {charger.power} kW active
                          </div>
                        )}
                      </div>
                      <div className="w-2 h-2 bg-popover border-b border-r border-border transform rotate-45 absolute left-1/2 -translate-x-1/2 -bottom-1"></div>
                    </div>
                  </div>
                </div>
              );
            })}
          </div>
        </div>

        <div className="grid grid-cols-2 md:grid-cols-4 gap-4 mt-4">
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full bg-primary"></div>
            <span className="text-sm">Charging</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full bg-idle"></div>
            <span className="text-sm">Available</span>
          </div>
          <div className="flex items-center gap-2">
            <div className="w-3 h-3 rounded-full bg-muted"></div>
            <span className="text-sm">Offline</span>
          </div>
          <div className="flex items-center gap-2">
            <DollarSign className="w-3 h-3 text-primary" />
            <span className="text-sm">Dynamic Pricing</span>
          </div>
        </div>

        {selectedCharger && (
          <div className="mt-4 p-4 bg-primary/10 border border-primary/20 rounded-lg" data-testid="selected-charger-details">
            <div className="flex items-start justify-between gap-4">
              <div className="flex-1">
                <h4 className="font-semibold">{selectedCharger.chargerName}</h4>
                <p className="text-sm text-muted-foreground">{selectedCharger.address}</p>
                <div className="flex items-center gap-4 mt-2 flex-wrap">
                  <div className="text-sm">
                    <span className="text-muted-foreground">Current Price: </span>
                    <span className="font-bold text-primary">${currentPrice(selectedCharger).toFixed(2)}/kWh</span>
                  </div>
                  <div className="text-sm">
                    <span className="text-muted-foreground">Max Power: </span>
                    <span className="font-semibold">{selectedCharger.maxPower} kW</span>
                  </div>
                </div>
                <div className="flex items-center gap-2 mt-2 flex-wrap">
                  <Badge variant="outline" className="text-xs">
                    {selectedCharger.connectorType}
                  </Badge>
                  {selectedCharger.v2gCapable && (
                    <Badge variant="default" className="text-xs" data-testid={`v2g-badge-selected-${selectedCharger.chargerId}`}>
                      <ArrowUpCircle className="h-3 w-3 mr-1" />
                      V2G Capable
                    </Badge>
                  )}
                </div>
              </div>
              <Button size="sm" data-testid="button-navigate">
                <Navigation className="h-4 w-4 mr-2" />
                Navigate
              </Button>
            </div>
          </div>
        )}
      </CardContent>
    </Card>
  );
}
