import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { CheckCircle2, Circle, Wrench } from "lucide-react";

interface InstallStep {
  id: string;
  label: string;
  completed: boolean;
}

export function InstallerPanel() {
  const [steps, setSteps] = useState<InstallStep[]>([
    { id: "network", label: "Network Configuration", completed: false },
    { id: "limits", label: "Set Current Limits", completed: false },
    { id: "tests", label: "Run Diagnostics", completed: false },
    { id: "connect", label: "Connect to Backend", completed: false },
    { id: "report", label: "Generate Report", completed: false },
  ]);

  const completedCount = steps.filter((s) => s.completed).length;
  const progress = (completedCount / steps.length) * 100;

  const toggleStep = (id: string) => {
    setSteps(steps.map((s) => (s.id === id ? { ...s, completed: !s.completed } : s)));
  };

  return (
    <Card data-testid="panel-installer">
      <CardHeader>
        <div className="flex items-center justify-between">
          <div className="flex items-center gap-2">
            <Wrench className="h-5 w-5 text-warning" />
            <CardTitle>Installer Mode</CardTitle>
          </div>
          <Badge variant="outline" className="bg-warning/20 text-warning">
            Active
          </Badge>
        </div>
      </CardHeader>
      <CardContent className="space-y-6">
        <div className="space-y-2">
          <div className="flex justify-between text-sm">
            <span className="font-medium">Installation Progress</span>
            <span className="text-muted-foreground">
              {completedCount} of {steps.length} steps
            </span>
          </div>
          <Progress value={progress} />
        </div>

        <div className="space-y-2">
          {steps.map((step) => (
            <div
              key={step.id}
              className="flex items-center gap-3 p-3 rounded-lg bg-card border border-card-border hover-elevate cursor-pointer"
              onClick={() => toggleStep(step.id)}
              data-testid={`step-${step.id}`}
            >
              {step.completed ? (
                <CheckCircle2 className="h-5 w-5 text-primary" />
              ) : (
                <Circle className="h-5 w-5 text-muted-foreground" />
              )}
              <span className={step.completed ? "text-muted-foreground" : ""}>
                {step.label}
              </span>
            </div>
          ))}
        </div>

        <div className="space-y-3 pt-4 border-t">
          <div className="space-y-2">
            <Label htmlFor="serial">Serial Number</Label>
            <Input
              id="serial"
              placeholder="FBOS-2024-XXXXX"
              className="font-mono"
              data-testid="input-serial"
            />
          </div>
          <div className="space-y-2">
            <Label htmlFor="max-current">Max Current (A)</Label>
            <Input
              id="max-current"
              type="number"
              placeholder="32"
              data-testid="input-max-current"
            />
          </div>
        </div>

        <Button className="w-full" data-testid="button-generate-report">
          Generate Install Report
        </Button>
      </CardContent>
    </Card>
  );
}
