import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { PowerGauge } from "./PowerGauge";
import { MetricDisplay } from "./MetricDisplay";
import { StatusBadge, type ChargerStatus } from "./StatusBadge";
import { ChargingModeSelector, type ChargingMode } from "./ChargingModeSelector";
import { Play, Square, Pause, Lock } from "lucide-react";

interface LiveChargingCardProps {
  chargerName: string;
  status: ChargerStatus;
  currentPower: number;
  maxPower: number;
  current: number;
  energyDispensed: number;
  duration: string;
  mode: ChargingMode;
  onModeChange?: (mode: ChargingMode) => void;
  onStart?: () => void;
  onStop?: () => void;
  onPause?: () => void;
  onLock?: () => void;
}

export function LiveChargingCard({
  chargerName,
  status,
  currentPower,
  maxPower,
  current,
  energyDispensed,
  duration,
  mode,
  onModeChange,
  onStart,
  onStop,
  onPause,
  onLock,
}: LiveChargingCardProps) {
  return (
    <Card data-testid="card-live-charging">
      <CardHeader>
        <div className="flex items-center justify-between">
          <CardTitle>{chargerName}</CardTitle>
          <StatusBadge status={status} />
        </div>
      </CardHeader>
      <CardContent className="space-y-6">
        <div className="flex justify-center">
          <PowerGauge
            currentPower={currentPower}
            maxPower={maxPower}
            mode={mode}
          />
        </div>

        <div className="grid grid-cols-3 gap-4">
          <MetricDisplay label="Current" value={current} unit="A" />
          <MetricDisplay label="Energy" value={energyDispensed.toFixed(1)} unit="kWh" />
          <MetricDisplay label="Duration" value={duration} />
        </div>

        <ChargingModeSelector
          value={mode}
          onChange={(m) => onModeChange?.(m)}
          disabled={status === "offline"}
        />

        <div className="flex gap-2">
          {status === "idle" || status === "paused" ? (
            <Button
              className="flex-1"
              onClick={onStart}
              data-testid="button-start"
            >
              <Play className="h-4 w-4 mr-2" />
              Start
            </Button>
          ) : (
            <Button
              className="flex-1"
              variant="outline"
              onClick={onPause}
              disabled={status === "offline"}
              data-testid="button-pause"
            >
              <Pause className="h-4 w-4 mr-2" />
              Pause
            </Button>
          )}
          <Button
            variant="destructive"
            onClick={onStop}
            disabled={status === "idle" || status === "offline"}
            data-testid="button-stop"
          >
            <Square className="h-4 w-4 mr-2" />
            Stop
          </Button>
          <Button
            variant="outline"
            size="icon"
            onClick={onLock}
            disabled={status === "offline"}
            data-testid="button-lock"
          >
            <Lock className="h-4 w-4" />
          </Button>
        </div>
      </CardContent>
    </Card>
  );
}
