import { cn } from "@/lib/utils";

interface PowerGaugeProps {
  currentPower: number;
  maxPower: number;
  mode?: "standard" | "eco" | "boost";
  className?: string;
}

export function PowerGauge({
  currentPower,
  maxPower,
  mode = "standard",
  className,
}: PowerGaugeProps) {
  const percentage = Math.min((currentPower / maxPower) * 100, 100);
  const circumference = 2 * Math.PI * 70;
  const strokeDashoffset = circumference - (percentage / 100) * circumference;

  const modeColors = {
    standard: "text-primary",
    eco: "text-warning",
    boost: "text-destructive",
  };

  return (
    <div className={cn("relative flex items-center justify-center", className)} data-testid="gauge-power">
      <svg className="w-48 h-48 -rotate-90" viewBox="0 0 160 160">
        <circle
          cx="80"
          cy="80"
          r="70"
          stroke="currentColor"
          strokeWidth="8"
          fill="none"
          className="text-muted/20"
        />
        <circle
          cx="80"
          cy="80"
          r="70"
          stroke="currentColor"
          strokeWidth="8"
          fill="none"
          strokeDasharray={circumference}
          strokeDashoffset={strokeDashoffset}
          strokeLinecap="round"
          className={cn("transition-all duration-300", modeColors[mode])}
        />
      </svg>
      <div className="absolute inset-0 flex flex-col items-center justify-center">
        <div className="text-4xl font-bold" data-testid="text-current-power">
          {currentPower.toFixed(1)}
        </div>
        <div className="text-sm text-muted-foreground">kW</div>
        <div className="text-xs text-muted-foreground mt-1">
          of {maxPower} kW
        </div>
      </div>
    </div>
  );
}
