import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { ChargerCard } from "@/components/ChargerCard";
import { ChargerMap } from "@/components/ChargerMap";
import { singaporeChargers, singaporeLocations } from "@shared/singapore-data";
import { MapPin, Plus } from "lucide-react";

const locationChargers = singaporeLocations.map(location => {
  const chargers = singaporeChargers
    .filter(charger => charger.id === location.id)
    .map(charger => ({
      id: charger.chargerId,
      name: charger.chargerName,
      status: charger.status,
      power: charger.power,
      lastActivity: charger.status === "charging" ? "Active now" : 
                     charger.status === "idle" ? "Available" : "Offline"
    }));
  
  return {
    id: location.id,
    name: location.name,
    address: location.address,
    chargers
  };
});

export default function Locations() {
  const [selectedCharger, setSelectedCharger] = useState(singaporeChargers[0]);

  return (
    <div className="space-y-6">
      <div className="flex items-center justify-between">
        <div>
          <h1 className="text-4xl font-bold">Locations</h1>
          <p className="text-muted-foreground mt-2">
            Manage chargers across Singapore with dynamic pricing
          </p>
        </div>
        <Button data-testid="button-add-location">
          <Plus className="h-4 w-4 mr-2" />
          Add Location
        </Button>
      </div>

      <ChargerMap
        chargers={singaporeChargers}
        selectedCharger={selectedCharger}
        onChargerSelect={setSelectedCharger}
      />

      <div className="space-y-6">
        {locationChargers.map((location) => (
          <Card key={location.id} data-testid={`location-${location.id}`}>
            <CardHeader>
              <div className="flex items-start justify-between">
                <div>
                  <CardTitle className="flex items-center gap-2">
                    <MapPin className="h-5 w-5 text-primary" />
                    {location.name}
                  </CardTitle>
                  <p className="text-sm text-muted-foreground mt-1">
                    {location.address}
                  </p>
                </div>
                <Button variant="outline" size="sm" data-testid={`button-manage-${location.id}`}>
                  Manage
                </Button>
              </div>
            </CardHeader>
            <CardContent>
              <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-4">
                {location.chargers.map((charger) => (
                  <ChargerCard
                    key={charger.id}
                    {...charger}
                    location={location.name}
                    onStart={() => console.log("Start", charger.id)}
                    onSettings={() => console.log("Settings", charger.id)}
                  />
                ))}
              </div>
            </CardContent>
          </Card>
        ))}
      </div>
    </div>
  );
}
