import { useState } from "react";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Progress } from "@/components/ui/progress";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { LoadManagementPanel } from "@/components/LoadManagementPanel";
import { InstallerPanel } from "@/components/InstallerPanel";
import {
  Settings,
  Zap,
  Thermometer,
  Battery,
  Cable,
  Shield,
  CheckCircle2,
  AlertTriangle,
  Info,
  TrendingUp,
  Plus,
  Wrench,
} from "lucide-react";

const technicalSpecs = [
  {
    category: "Charging Rates",
    specs: [
      { name: "AC Level 2", value: "7.4 kW - 22 kW", unit: "230V, 32-100A", status: "standard" },
      { name: "DC Fast Charging", value: "50 kW - 350 kW", unit: "CCS2 Standard", status: "available" },
      { name: "Ultra-Fast DC", value: "150 kW - 350 kW", unit: "Liquid Cooled", status: "premium" },
    ],
  },
  {
    category: "Connection Types",
    specs: [
      { name: "Type 2 (IEC 62196)", value: "AC Charging", unit: "Up to 43 kW", status: "standard" },
      { name: "CCS2 Combo", value: "DC Fast Charging", unit: "Up to 350 kW", status: "standard" },
      { name: "CHAdeMO", value: "Legacy DC", unit: "Up to 62.5 kW", status: "legacy" },
    ],
  },
  {
    category: "Thermal Management",
    specs: [
      { name: "Active Cooling", value: "Liquid-cooled cables", unit: "150+ kW stations", status: "active" },
      { name: "Temperature Monitoring", value: "Real-time sensors", unit: "All stations", status: "active" },
      { name: "Thermal Throttling", value: "Auto power reduction", unit: "> 45°C", status: "enabled" },
    ],
  },
];

const recentInstallations = [
  { id: "1", location: "Jurong East Hub", type: "DC Fast Charger", power: "150 kW", date: "2 days ago", status: "commissioned" },
  { id: "2", location: "Marina Bay Complex", type: "Ultra-Fast DC", power: "350 kW", date: "5 days ago", status: "commissioned" },
  { id: "3", location: "Woodlands Industrial", type: "AC Level 2", power: "22 kW", date: "1 week ago", status: "commissioned" },
  { id: "4", location: "Tampines Mall", type: "DC Fast Charger", power: "150 kW", date: "2 weeks ago", status: "commissioned" },
];

const batteryWarranties = [
  { manufacturer: "Tesla", model: "Model 3/Y", coverage: "8 years / 192,000 km", degradation: "70% capacity", fastChargeLimit: "No limit" },
  { manufacturer: "BYD", model: "Atto 3", coverage: "8 years / 150,000 km", degradation: "70% capacity", fastChargeLimit: "Daily recommended" },
  { manufacturer: "Hyundai", model: "Ioniq 5/6", coverage: "8 years / 160,000 km", degradation: "70% capacity", fastChargeLimit: "No limit" },
  { manufacturer: "Nissan", model: "Leaf", coverage: "8 years / 160,000 km", degradation: "75% capacity", fastChargeLimit: "2x/day max" },
];

const gridActiveRules = [
  { id: "1", name: "Peak Demand Response", condition: "Grid load > 85%", action: "Reduce charging rate by 30%", active: true },
  { id: "2", name: "Off-Peak Incentive", condition: "23:00 - 07:00", action: "Enable max power charging", active: true },
  { id: "3", name: "V2G Export Trigger", condition: "Grid price > $0.50/kWh", action: "Allow battery export", active: true },
  { id: "4", name: "Renewable Priority", condition: "Solar generation > 60%", action: "Prioritize charging", active: true },
];

const loadManagementRules = [
  { id: "1", name: "Building Capacity Limit", condition: "Total load > 200A", action: "Balance across chargers", active: true },
  { id: "2", name: "Transformer Protection", condition: "Load > 80% capacity", action: "Limit to 16A per charger", active: true },
  { id: "3", name: "Time-based Distribution", condition: "Peak hours (18-22)", action: "Rotate charging slots", active: true },
];

export default function Technical() {
  const [selectedTab, setSelectedTab] = useState("specs");

  // Session limits constants
  const MAX_SESSIONS_DISPLAY = 5;
  const MAX_SESSIONS_PER_CAR = 12;

  return (
    <div className="space-y-6">
      <div>
        <h1 className="text-4xl font-bold">Technical</h1>
        <p className="text-muted-foreground mt-2">
          EV charger technical specifications, installation, and grid management
        </p>
      </div>

      {/* Session Limits Info Card */}
      <Card className="border-primary/50 bg-primary/5" data-testid="card-session-limits">
        <CardContent className="pt-6">
          <div className="flex items-start gap-4">
            <div className="rounded-full bg-primary/20 p-2">
              <Info className="h-5 w-5 text-primary" />
            </div>
            <div className="flex-1">
              <h3 className="font-semibold mb-2">Session Limits</h3>
              <div className="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div className="flex items-center gap-2">
                  <Badge variant="outline" className="bg-background">
                    Max Sessions per Car
                  </Badge>
                  <span className="text-2xl font-bold text-primary" data-testid="text-max-sessions-per-car">
                    {MAX_SESSIONS_PER_CAR}
                  </span>
                </div>
                <div className="flex items-center gap-2">
                  <Badge variant="outline" className="bg-background">
                    Max Sessions Displayed
                  </Badge>
                  <span className="text-2xl font-bold text-primary" data-testid="text-max-sessions-display">
                    {MAX_SESSIONS_DISPLAY}
                  </span>
                </div>
              </div>
              <p className="text-sm text-muted-foreground mt-3">
                Platform displays maximum {MAX_SESSIONS_DISPLAY} charging sessions at once. Each car can have up to {MAX_SESSIONS_PER_CAR} active sessions.
              </p>
            </div>
          </div>
        </CardContent>
      </Card>

      <Tabs value={selectedTab} onValueChange={setSelectedTab} className="space-y-6">
        <TabsList className="grid w-full grid-cols-4">
          <TabsTrigger value="specs" data-testid="tab-specs">
            <Zap className="h-4 w-4 mr-2" />
            Charger Specs
          </TabsTrigger>
          <TabsTrigger value="warranties" data-testid="tab-warranties">
            <Shield className="h-4 w-4 mr-2" />
            Warranties & Rules
          </TabsTrigger>
          <TabsTrigger value="load" data-testid="tab-load">
            <Settings className="h-4 w-4 mr-2" />
            Load Management
          </TabsTrigger>
          <TabsTrigger value="installer" data-testid="tab-installer">
            <Wrench className="h-4 w-4 mr-2" />
            Installer Mode
          </TabsTrigger>
        </TabsList>

        {/* Charger Specifications Tab */}
        <TabsContent value="specs" className="space-y-6">
          <div className="grid grid-cols-1 lg:grid-cols-2 xl:grid-cols-3 gap-6">
            {technicalSpecs.map((category) => (
              <Card key={category.category} data-testid={`card-${category.category.toLowerCase().replace(/\s/g, '-')}`}>
                <CardHeader>
                  <CardTitle className="flex items-center gap-2">
                    {category.category === "Charging Rates" && <TrendingUp className="h-5 w-5 text-primary" />}
                    {category.category === "Connection Types" && <Cable className="h-5 w-5 text-primary" />}
                    {category.category === "Thermal Management" && <Thermometer className="h-5 w-5 text-primary" />}
                    {category.category}
                  </CardTitle>
                </CardHeader>
                <CardContent className="space-y-3">
                  {category.specs.map((spec) => (
                    <div key={spec.name} className="space-y-1">
                      <div className="flex items-center justify-between">
                        <span className="text-sm font-medium">{spec.name}</span>
                        <Badge
                          variant="outline"
                          className={
                            spec.status === "active" || spec.status === "standard" || spec.status === "enabled"
                              ? "bg-primary/20 text-primary"
                              : spec.status === "available" || spec.status === "premium"
                              ? "bg-accent/20 text-accent-foreground"
                              : "bg-muted"
                          }
                        >
                          {spec.status}
                        </Badge>
                      </div>
                      <div className="text-sm text-muted-foreground">
                        <div className="font-mono font-semibold text-foreground">{spec.value}</div>
                        <div>{spec.unit}</div>
                      </div>
                    </div>
                  ))}
                </CardContent>
              </Card>
            ))}
          </div>

          {/* Recent Fast Charger Installations */}
          <Card data-testid="card-recent-installations">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <CheckCircle2 className="h-5 w-5 text-primary" />
                Recent Fast Charger Installations
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {recentInstallations.map((install) => (
                  <div
                    key={install.id}
                    className="flex items-center justify-between p-3 rounded-lg bg-card border border-card-border"
                    data-testid={`installation-${install.id}`}
                  >
                    <div className="flex-1">
                      <div className="font-medium">{install.location}</div>
                      <div className="text-sm text-muted-foreground">
                        {install.type} • {install.power} • {install.date}
                      </div>
                    </div>
                    <Badge variant="outline" className="bg-primary/20 text-primary">
                      {install.status}
                    </Badge>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Warranties & Rules Tab */}
        <TabsContent value="warranties" className="space-y-6">
          {/* Battery Warranties */}
          <Card data-testid="card-battery-warranties">
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                <Battery className="h-5 w-5 text-primary" />
                EV Battery Warranties & Fast Charging Limits
              </CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-3">
                {batteryWarranties.map((warranty) => (
                  <div
                    key={warranty.model}
                    className="p-4 rounded-lg bg-card border border-card-border"
                    data-testid={`warranty-${warranty.manufacturer.toLowerCase()}`}
                  >
                    <div className="flex items-start justify-between mb-3">
                      <div>
                        <div className="font-semibold">{warranty.manufacturer} {warranty.model}</div>
                        <div className="text-sm text-muted-foreground">{warranty.coverage}</div>
                      </div>
                      <Badge variant="outline" className="bg-primary/20">
                        {warranty.degradation}
                      </Badge>
                    </div>
                    <div className="flex items-center gap-2 text-sm">
                      {warranty.fastChargeLimit === "No limit" ? (
                        <CheckCircle2 className="h-4 w-4 text-primary" />
                      ) : (
                        <AlertTriangle className="h-4 w-4 text-warning" />
                      )}
                      <span className="text-muted-foreground">Fast Charge Limit:</span>
                      <span className="font-medium">{warranty.fastChargeLimit}</span>
                    </div>
                  </div>
                ))}
              </div>
            </CardContent>
          </Card>

          {/* Grid Active Rules */}
          <Card data-testid="card-grid-rules">
            <CardHeader>
              <div className="flex items-center justify-between">
                <CardTitle className="flex items-center gap-2">
                  <Zap className="h-5 w-5 text-primary" />
                  Grid Active Rules
                </CardTitle>
                <Button size="sm" data-testid="button-add-grid-rule">
                  <Plus className="h-4 w-4 mr-2" />
                  Add Rule
                </Button>
              </div>
            </CardHeader>
            <CardContent className="space-y-3">
              {gridActiveRules.map((rule) => (
                <div
                  key={rule.id}
                  className="flex items-start justify-between p-3 rounded-lg bg-card border border-card-border"
                  data-testid={`grid-rule-${rule.id}`}
                >
                  <div className="flex-1">
                    <div className="flex items-center gap-2 mb-1">
                      <div className="font-medium">{rule.name}</div>
                      <Badge
                        variant={rule.active ? "default" : "outline"}
                        className={rule.active ? "bg-primary/20" : ""}
                      >
                        {rule.active ? "Active" : "Inactive"}
                      </Badge>
                    </div>
                    <div className="text-sm text-muted-foreground">
                      <span className="font-medium">When:</span> {rule.condition}
                    </div>
                    <div className="text-sm text-muted-foreground">
                      <span className="font-medium">Then:</span> {rule.action}
                    </div>
                  </div>
                  <Button variant="ghost" size="icon" className="h-8 w-8">
                    <Settings className="h-4 w-4" />
                  </Button>
                </div>
              ))}
            </CardContent>
          </Card>
        </TabsContent>

        {/* Load Management Tab */}
        <TabsContent value="load" className="space-y-6">
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <LoadManagementPanel />

            <Card data-testid="card-load-rules">
              <CardHeader>
                <div className="flex items-center justify-between">
                  <CardTitle>Load Management Rules</CardTitle>
                  <Button size="sm" data-testid="button-add-load-rule">
                    <Plus className="h-4 w-4 mr-2" />
                    Add Rule
                  </Button>
                </div>
              </CardHeader>
              <CardContent className="space-y-3">
                {loadManagementRules.map((rule) => (
                  <div
                    key={rule.id}
                    className="flex items-start justify-between p-3 rounded-lg bg-card border border-card-border"
                    data-testid={`load-rule-${rule.id}`}
                  >
                    <div className="flex-1">
                      <div className="flex items-center gap-2 mb-1">
                        <div className="font-medium">{rule.name}</div>
                        <Badge
                          variant={rule.active ? "default" : "outline"}
                          className={rule.active ? "bg-primary/20" : ""}
                        >
                          {rule.active ? "Active" : "Inactive"}
                        </Badge>
                      </div>
                      <div className="text-sm text-muted-foreground">
                        <span className="font-medium">When:</span> {rule.condition}
                      </div>
                      <div className="text-sm text-muted-foreground">
                        <span className="font-medium">Then:</span> {rule.action}
                      </div>
                    </div>
                    <Button variant="ghost" size="icon" className="h-8 w-8">
                      <Settings className="h-4 w-4" />
                    </Button>
                  </div>
                ))}
              </CardContent>
            </Card>
          </div>

          <Card>
            <CardHeader>
              <CardTitle>Load Management History</CardTitle>
            </CardHeader>
            <CardContent>
              <div className="space-y-4">
                <div className="flex items-center justify-between p-3 bg-muted/50 rounded-lg">
                  <div>
                    <div className="font-medium">Transformer Protection Triggered</div>
                    <div className="text-sm text-muted-foreground">Today 19:15 - Load exceeded 80% capacity</div>
                  </div>
                  <div className="text-sm text-warning font-medium">Limited to 16A</div>
                </div>
                <div className="flex items-center justify-between p-3 bg-muted/50 rounded-lg">
                  <div>
                    <div className="font-medium">Building Capacity Balance</div>
                    <div className="text-sm text-muted-foreground">Today 18:30 - Total load exceeded 200A</div>
                  </div>
                  <div className="text-sm text-primary font-medium">Balanced</div>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        {/* Installer Mode Tab */}
        <TabsContent value="installer" className="space-y-6">
          <div className="grid grid-cols-1 xl:grid-cols-2 gap-6">
            <InstallerPanel />

            <Card data-testid="card-installer-diagnostics">
              <CardHeader>
                <CardTitle>Live Diagnostics</CardTitle>
              </CardHeader>
              <CardContent className="space-y-4">
                <div className="grid grid-cols-2 gap-4">
                  <div className="space-y-1">
                    <div className="text-xs uppercase tracking-wide text-muted-foreground">Phase L1</div>
                    <div className="text-2xl font-bold font-mono">238.4V</div>
                  </div>
                  <div className="space-y-1">
                    <div className="text-xs uppercase tracking-wide text-muted-foreground">Phase L2</div>
                    <div className="text-2xl font-bold font-mono">239.1V</div>
                  </div>
                  <div className="space-y-1">
                    <div className="text-xs uppercase tracking-wide text-muted-foreground">Phase L3</div>
                    <div className="text-2xl font-bold font-mono">237.8V</div>
                  </div>
                  <div className="space-y-1">
                    <div className="text-xs uppercase tracking-wide text-muted-foreground">Temperature</div>
                    <div className="text-2xl font-bold font-mono">42°C</div>
                  </div>
                </div>

                <div className="pt-4 border-t space-y-3">
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <Cable className="h-4 w-4 text-primary" />
                      <span className="text-sm font-medium">Network Connection</span>
                    </div>
                    <Badge variant="outline" className="bg-primary/20 text-primary">
                      Connected
                    </Badge>
                  </div>
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <Zap className="h-4 w-4 text-primary" />
                      <span className="text-sm font-medium">Contactor Status</span>
                    </div>
                    <Badge variant="outline" className="bg-primary/20 text-primary">
                      OK
                    </Badge>
                  </div>
                  <div className="flex items-center justify-between">
                    <div className="flex items-center gap-2">
                      <Thermometer className="h-4 w-4 text-primary" />
                      <span className="text-sm font-medium">Thermal Management</span>
                    </div>
                    <Badge variant="outline" className="bg-primary/20 text-primary">
                      Normal
                    </Badge>
                  </div>
                </div>

                <div className="pt-4 border-t">
                  <Button variant="outline" className="w-full" data-testid="button-run-diagnostics">
                    <Wrench className="h-4 w-4 mr-2" />
                    Run Full Diagnostic Suite
                  </Button>
                </div>
              </CardContent>
            </Card>
          </div>
        </TabsContent>
      </Tabs>
    </div>
  );
}
